package awacs

import (
	"encoding/json"
	"fmt"
	"sort"
	"strings"
	"time"

	awacsactivities "a.yandex-team.ru/infra/temporal/activities/awacs"
	"go.temporal.io/sdk/temporal"
	"go.temporal.io/sdk/workflow"

	awacspb "a.yandex-team.ru/infra/awacs/proto"
)

func MigrateBackendToSDWorkflow(ctx workflow.Context, namespaceID string) (string, error) {
	origCtx := ctx

	ctx = workflow.WithActivityOptions(origCtx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 30,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.0,
			MaximumInterval:        time.Minute,
			MaximumAttempts:        10,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
		HeartbeatTimeout: time.Minute * 2,
	})

	logger := workflow.GetLogger(ctx)

	var act *awacsactivities.Activities

	waitUntilNamespaceIsUnpausedAndSettledCtx := workflow.WithActivityOptions(origCtx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 5,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:    time.Minute,
			BackoffCoefficient: 1.0,
			MaximumInterval:    time.Minute,
			MaximumAttempts:    60,
		},
	})
	if err := workflow.ExecuteActivity(
		waitUntilNamespaceIsUnpausedAndSettledCtx,
		act.WaitUntilNamespaceIsUnpausedAndSettled,
		namespaceID,
	).Get(waitUntilNamespaceIsUnpausedAndSettledCtx, nil); err != nil {
		return "", err
	}

	var backends []*awacspb.Backend
	fut := workflow.ExecuteActivity(ctx, act.ListBackends, namespaceID)
	err := fut.Get(ctx, &backends)
	if err != nil {
		return "", fmt.Errorf("failed to list backends for namespace %s: %w", namespaceID, err)
	}

	var aspectsSetRsp awacspb.GetNamespaceAspectsSetResponse
	if err := workflow.ExecuteActivity(ctx, act.GetNamespaceAspectsSet, namespaceID).Get(ctx, &aspectsSetRsp); err != nil {
		return "", err
	}
	aspectsSet := aspectsSetRsp.AspectsSet
	backendGroups, err := getBackendGroups(workflow.Now(ctx), namespaceID, aspectsSet.Content.Graph)
	if err != nil {
		return "", err
	}

	var rpsStats []awacsactivities.NamespaceRps
	if err := workflow.ExecuteActivity(ctx, act.GetYesterdayMaxRpsStatsByNamespace).Get(ctx, &rpsStats); err != nil {
		return "", err
	}

	var namespaceRps float32
	for i := range rpsStats {
		if rpsStats[i].NamespaceID == namespaceID {
			namespaceRps = rpsStats[i].Rps
			break
		}
	}

	l7BackendIds := make(map[string]struct{})
	unusedBackendIds := make(map[string]struct{})
	l7BackendIdsToBeMigrated := make(map[string]struct{})
	l7BackendIDTypes := make(map[string]string)

	for _, backendPb := range backends {
		backendID := backendPb.Meta.Id

		if backendPb.Spec.Deleted {
			continue
		}
		if backendPb.ResolverStatus == nil ||
			backendPb.ResolverStatus.LastAttempt == nil ||
			backendPb.ResolverStatus.LastAttempt.Succeeded.Status != "True" {

			continue
		}
		usedInDNS := false
		for _, revPb := range backendPb.DnsRecordStatuses {
			for _, condPb := range revPb.Validated {
				if condPb.Status == "True" {
					usedInDNS = true
				}
			}
		}

		usedInL3 := false
		for _, revPb := range backendPb.L3Statuses {
			for _, condPb := range revPb.Validated {
				if condPb.Status == "True" {
					usedInL3 = true
				}
			}
		}

		usedInL7 := false
		for _, revPb := range backendPb.Statuses {
			for _, condPb := range revPb.Validated {
				if condPb.Status == "True" {
					usedInL7 = true
					l7BackendIDTypes[backendID] = backendPb.Spec.Selector.Type.String()
				}
			}
		}

		if usedInL7 {
			l7BackendIds[backendID] = struct{}{}
		}

		if !usedInDNS && !usedInL3 && !usedInL7 {
			unusedBackendIds[backendID] = struct{}{}
		}

		isBackendToBeMigrated := false
		if !usedInDNS && !usedInL3 &&
			backendPb.Spec.Selector.Type == awacspb.BackendSelector_YP_ENDPOINT_SETS &&
			(backendPb.Spec.Selector.Port == nil || backendPb.Spec.Selector.Port.Policy == awacspb.Port_KEEP) {

			isBackendToBeMigrated = true
			for _, esPb := range backendPb.Spec.Selector.YpEndpointSets {
				fut := workflow.ExecuteActivity(ctx, act.DoesEndpointsetExist, esPb.Cluster, esPb.EndpointSetId)
				var endpointsetExists bool
				if err := fut.Get(ctx, &endpointsetExists); err != nil {
					return "", err
				}

				if (esPb.Port != nil && esPb.Port.Policy != awacspb.Port_KEEP ||
					esPb.Weight != nil && esPb.Weight.Policy != awacspb.Weight_KEEP) ||
					!endpointsetExists {
					isBackendToBeMigrated = false
				}
			}
		} else {
			isBackendToBeMigrated = false
		}

		if isBackendToBeMigrated {
			l7BackendIdsToBeMigrated[backendID] = struct{}{}
		}
	}

	for _, grpBackendIds := range backendGroups {
		if len(setIntersectionString(grpBackendIds, l7BackendIdsToBeMigrated)) > 0 &&
			!isSubsetString(l7BackendIdsToBeMigrated, grpBackendIds) {
			msg := fmt.Sprintf(
				"group [%s] contains both fixable and unfixable backends",
				strings.Join(setToSliceString(grpBackendIds), ", "))
			logger.Info(msg)
			l7BackendIdsToBeMigrated = setDifferenceString(l7BackendIdsToBeMigrated, grpBackendIds)
		}
	}

	l7BackendIdsToBeMigrated = setDifferenceString(l7BackendIdsToBeMigrated, unusedBackendIds)

	if len(l7BackendIdsToBeMigrated) > 0 {
		totalNonSd := 0
		for _, v := range l7BackendIDTypes {
			if v != "YP_ENDPOINT_SETS_SD" {
				totalNonSd += 1
			}
		}

		logger.Info(
			"total non-SD backends:", totalNonSd,
			", to be migrated:", len(l7BackendIdsToBeMigrated))

		ids := make([]string, 0, len(l7BackendIdsToBeMigrated))
		for backendID := range l7BackendIdsToBeMigrated {
			ids = append(ids, backendID)
		}

		sort.Strings(ids)
		logger.Info(fmt.Sprintf("Backends to be migrated: [%s]", strings.Join(ids, ", ")))
		logger.Info(fmt.Sprintf("Namespace %s serves %f RPS at peak times", namespaceID, namespaceRps))

		var balancers []*awacspb.Balancer
		if err := workflow.ExecuteActivity(ctx, act.ListBalancers, namespaceID).Get(ctx, &balancers); err != nil {
			return "", err
		}

		sort.Slice(balancers, func(i, j int) bool {
			return balancers[i].Meta.Id < balancers[j].Meta.Id
		})

		for _, balancer := range balancers {
			if err := workflow.ExecuteActivity(
				ctx, act.PauseBalancer, namespaceID, balancer.Meta.Id).Get(ctx, nil); err != nil {
				return "", err
			}
		}

		logger.Info(fmt.Sprintf("The tool is about to migrate %v affected backends in %s namespace",
			len(l7BackendIdsToBeMigrated), namespaceID))

		updatedBackendRevisions := make([]string, 0, len(backends))
		for _, backendPb := range backends {
			if _, found := l7BackendIdsToBeMigrated[backendPb.Meta.Id]; !found {
				continue
			}
			var resp *awacspb.UpdateBackendResponse
			backendPb.Spec.Selector.Type = awacspb.BackendSelector_YP_ENDPOINT_SETS_SD
			if err := workflow.ExecuteActivity(
				ctx,
				act.UpdateBackend,
				&backendPb.Spec,
				backendPb.Meta.NamespaceId,
				backendPb.Meta.Id,
				backendPb.Meta.Version,
				"AWACS-484: use SD",
			).Get(ctx, &resp); err != nil {
				return "", err
			}
			updatedBackendRevisions = append(updatedBackendRevisions, resp.Backend.Meta.Version)
			logger.Info("Updated backend", backendPb.Meta.Id)
		}

		unpausedBalancers := uint(0)
		for _, balancer := range balancers {
			if err := workflow.ExecuteActivity(
				ctx, act.UnpauseBalancer, namespaceID, balancer.Meta.Id).Get(ctx, nil); err != nil {
				return "", err
			}

			unpausedBalancers += 1

			CheckBackendsRevisionsStatusesCtx := workflow.WithActivityOptions(origCtx, workflow.ActivityOptions{
				ScheduleToCloseTimeout: time.Hour,
				HeartbeatTimeout:       time.Minute,
				RetryPolicy: &temporal.RetryPolicy{
					InitialInterval:        time.Minute,
					BackoffCoefficient:     1,
					MaximumInterval:        time.Minute,
					MaximumAttempts:        5,
					NonRetryableErrorTypes: []string{"PermanentError"},
				},
			})

			if err := workflow.ExecuteActivity(
				CheckBackendsRevisionsStatusesCtx, act.WaitUntilBackendsRevisionsStatusesActive,
				namespaceID, updatedBackendRevisions, unpausedBalancers,
			).Get(CheckBackendsRevisionsStatusesCtx, nil); err != nil {
				return "", err
			}
		}
	} else {
		msg := "No backends to be migrated"
		logger.Info(msg)
		return msg, nil
	}

	return "", nil
}

func getBackendGroups(now time.Time, namespaceID string, graphPb *awacspb.NamespaceGraphAspects) ([]map[string]struct{}, error) {
	ok := true
	lastAttemptPb := graphPb.Status.LastAttempt
	if lastAttemptPb.Succeeded.Status != "True" {
		ok = false
	}

	diff := now.Sub(lastAttemptPb.FinishedAt.AsTime())
	if diff > 9*time.Hour {
		ok = false
	}

	if !ok {
		return nil, fmt.Errorf("inclusion graph for %s is not OK", namespaceID)
	}

	rv := make([]map[string]struct{}, 0)

	var graph awacsactivities.InclusionGraph
	if err := json.Unmarshal([]byte(graphPb.Content.InclusionGraphJson), &graph); err != nil {
		return rv, err
	}

	for _, node := range graph {
		if node.Type == "upstream" {
			includedBackendIds := make(map[string]struct{})
			skipUpstream := false
			for _, flatID := range node.IncludedBackendIds {
				parts := strings.Split(flatID, "/")
				backendNamespaceID := parts[0]
				backendID := parts[1]
				if backendNamespaceID != namespaceID {
					skipUpstream = true
				}
				includedBackendIds[backendID] = struct{}{}
			}
			if !skipUpstream {
				rv = append(rv, includedBackendIds)
			}
		}
	}

	return rv, nil
}

func setToSliceString(s map[string]struct{}) []string {
	rv := make([]string, 0, len(s))
	for k := range s {
		rv = append(rv, k)
	}
	return rv
}

func setIntersectionString(s1, s2 map[string]struct{}) map[string]struct{} {
	res := make(map[string]struct{})
	for k1 := range s1 {
		if _, found := s2[k1]; found {
			res[k1] = struct{}{}
		}
	}
	return res
}

func isSubsetString(haystack, needle map[string]struct{}) bool {
	for k := range needle {
		if _, found := haystack[k]; !found {
			return false
		}
	}
	return true
}

func setDifferenceString(a, b map[string]struct{}) map[string]struct{} {
	res := make(map[string]struct{})
	for k := range a {
		if v, found := b[k]; !found {
			res[k] = v
		}
	}
	return res
}
