package awacs

import (
	"fmt"
	"time"

	enumspb "go.temporal.io/api/enums/v1"

	"go.temporal.io/sdk/temporal"
	"go.temporal.io/sdk/workflow"

	awacsactivities "a.yandex-team.ru/infra/temporal/activities/awacs"
)

const maxNamespacesCount = 1000

func CronNamespacesWorkflow(ctx workflow.Context) error {
	// Why we have different CronNamespacesWorkflow and CreateNamespaceWorkflowsWorkflow: https://wiki.yandex-team.ru/awacs/development/temporal/#cron
	cwo := workflow.ChildWorkflowOptions{WorkflowID: "create-namespace-workflows"}
	ctx = workflow.WithChildOptions(ctx, cwo)
	err := workflow.ExecuteChildWorkflow(ctx, CreateNamespaceWorkflowsWorkflow, 0).Get(ctx, nil)
	return err
}

func CreateNamespaceWorkflowsWorkflow(ctx workflow.Context, offset int) error {
	options := workflow.ActivityOptions{
		StartToCloseTimeout: time.Hour,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.0,
			MaximumAttempts:        10,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	}
	ctx = workflow.WithActivityOptions(ctx, options)

	var a *awacsactivities.Activities
	var namespaceIDs []string
	err := workflow.ExecuteActivity(ctx, a.ListNamespaceIds).Get(ctx, &namespaceIDs)
	if err != nil {
		return err
	}
	if offset >= len(namespaceIDs) {
		return nil
	}

	i := 0
	for _, namespaceID := range namespaceIDs[offset:] {
		cwo := workflow.ChildWorkflowOptions{
			WorkflowID:          fmt.Sprintf("main-namespace-processor(%s)", namespaceID),
			ParentClosePolicy:   enumspb.PARENT_CLOSE_POLICY_ABANDON,
			WorkflowTaskTimeout: time.Hour,
			TaskQueue:           "awacs-watching",
		}
		cwCtx := workflow.WithChildOptions(ctx, cwo)
		err = workflow.ExecuteChildWorkflow(cwCtx, ProcessNamespaceWorkflow, namespaceID).GetChildWorkflowExecution().Get(cwCtx, nil)
		if err != nil && !temporal.IsWorkflowExecutionAlreadyStartedError(err) {
			return err
		}

		i++
		if i >= maxNamespacesCount {
			// https://wiki.yandex-team.ru/awacs/development/temporal/#continue-as-new
			return workflow.NewContinueAsNewError(ctx, CreateNamespaceWorkflowsWorkflow, offset+i)
		}
	}
	return nil
}
