package awacs

import (
	"fmt"
	"os"
	"time"

	"go.temporal.io/sdk/temporal"

	awacsactivities "a.yandex-team.ru/infra/temporal/activities/awacs"
	swatactivities "a.yandex-team.ru/infra/temporal/activities/swat"
	"a.yandex-team.ru/infra/temporal/clients/startrek"

	enumspb "go.temporal.io/api/enums/v1"
	"go.temporal.io/sdk/workflow"

	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
)

const (
	InvocationText              = "Обратите внимание на тикет - ваши балансеры не могут активироваться."
	StartrekerNamespaceIDEnvVar = "STARTREKER_NAMESPACE_ID"
	WikiPathTemplate            = "startreker/templates/%s"
	AwacsAbcScheduleID          = 4645
)

type StuckTemplateData struct {
	NamespaceID string
	BalancerIds []string
}

func CreateOrUpdateStuckProblem(ctx workflow.Context, namespaceID, templateName, childWfID string, stuckBalancerIds []string, childWf workflow.ChildWorkflowFuture) (workflow.ChildWorkflowFuture, error) {
	var a *awacsactivities.Activities
	var s *swatactivities.Activities

	ctx = workflow.WithActivityOptions(ctx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 1,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.5,
			MaximumInterval:        time.Minute * 10,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	})

	startrekerNamespaceID := os.Getenv(StartrekerNamespaceIDEnvVar)
	if startrekerNamespaceID == "" {
		return nil, fmt.Errorf("startreker namespace ID must be set in %s env var", StartrekerNamespaceIDEnvVar)
	}
	if len(stuckBalancerIds) == 0 {
		if childWf != nil {
			// Child workflow has been created, need to close ticket
			childCtx := workflow.WithWorkflowNamespace(ctx, startrekerNamespaceID)
			childWf.SignalChildWorkflow(childCtx, "CloseProblem", nil)
		}
		return nil, nil
	}

	// Extract responsible
	v := workflow.GetVersion(ctx, "AWACS-1322", workflow.DefaultVersion, 1)
	var err error
	responsible := &processor.Responsible{}
	if v == workflow.DefaultVersion {
		var logins []string
		err = workflow.ExecuteActivity(ctx, a.ExtractResponsibleLogins, namespaceID).Get(ctx, &logins)
		if err != nil {
			return nil, err
		}
		responsible.Logins = logins
		responsible.Kind = processor.Logins
	} else {
		err = workflow.ExecuteActivity(ctx, a.ExtractResponsible, namespaceID).Get(ctx, &responsible)
		if err != nil {
			return nil, err
		}
	}
	if len(responsible.Logins) > 5 {
		responsible.Logins = responsible.Logins[:5]
	}

	// Create ticket summary/description
	templateData := StuckTemplateData{
		NamespaceID: namespaceID,
		BalancerIds: stuckBalancerIds,
	}
	var renderedTicket swatactivities.RenderedTicket
	err = workflow.ExecuteActivity(ctx, s.RenderTicketTemplate, fmt.Sprintf(WikiPathTemplate, templateName), templateData).Get(ctx, &renderedTicket)
	if err != nil {
		return nil, err
	}

	tags := []string{templateName}
	for _, balancerID := range stuckBalancerIds {
		tags = append(tags, fmt.Sprintf("balancer:%s:%s", namespaceID, balancerID))
	}

	problem := processor.Problem{
		Ticket: startrek.Ticket{
			Summary:     renderedTicket.Summary,
			Description: renderedTicket.Description,
			Queue:       &startrek.Queue{Key: "AWACSNOTIFY"},
			Tags:        tags,
		},
		InvocationSettings: processor.InvocationSettings{
			Responsible: *responsible,
			RetryInvocationSettings: processor.RetryInvocationSettings{
				Kind:   processor.Period,
				Period: time.Hour * 12,
			},
			Text: InvocationText,
		},
		InfraDutyInvocationSettings: &processor.InfraDutyInvocationSettings{
			AbcScheduleID: AwacsAbcScheduleID,
			Text:          "Привет!\nНужна твоя помощь, как дежурного awacs",
		},
	}

	// Create/update ticket workflow
	if childWf == nil {
		cwo := workflow.ChildWorkflowOptions{
			Namespace:         startrekerNamespaceID,
			WorkflowID:        childWfID,
			ParentClosePolicy: enumspb.PARENT_CLOSE_POLICY_ABANDON,
			TaskQueue:         "startreker",
		}
		childCtx := workflow.WithChildOptions(ctx, cwo)
		childWf = workflow.ExecuteChildWorkflow(childCtx, processor.ProcessWorkflow, problem)
		childWf.GetChildWorkflowExecution()
	} else {
		problem.InvocationSettings.Text = ""
		childCtx := workflow.WithWorkflowNamespace(ctx, startrekerNamespaceID)
		err = childWf.SignalChildWorkflow(childCtx, "UpdateProblem", processor.UpdateProblemSignal{Problem: problem, Notify: true}).Get(ctx, nil)
		if err != nil {
			return nil, err
		}
	}
	return childWf, nil
}
