package common

import (
	"errors"
	"fmt"

	"go.temporal.io/sdk/workflow"
)

type ChildWorkflowPool struct {
	size           int
	ctx            workflow.Context
	selector       workflow.Selector
	pendingFutures map[string]workflow.ChildWorkflowFuture
	ChildErrors    map[string]error
}

func NewChildWorkflowPool(ctx workflow.Context, size int) *ChildWorkflowPool {
	return &ChildWorkflowPool{
		size:           size,
		ctx:            ctx,
		selector:       workflow.NewSelector(ctx),
		pendingFutures: make(map[string]workflow.ChildWorkflowFuture),
		ChildErrors:    make(map[string]error),
	}
}

func (p *ChildWorkflowPool) ExecuteChildWorkflow(cwo workflow.ChildWorkflowOptions, childWorkflow interface{}, args ...interface{}) (workflow.ChildWorkflowFuture, error) {
	if len(p.pendingFutures) >= p.size {
		p.selector.Select(p.ctx)
	}
	ctx := workflow.WithChildOptions(p.ctx, cwo)
	future := workflow.ExecuteChildWorkflow(ctx, childWorkflow, args...)

	var childWE workflow.Execution
	if err := future.GetChildWorkflowExecution().Get(ctx, &childWE); err != nil {
		return nil, fmt.Errorf("failed to execute child workflow: %w", err)
	}

	childWorkflowRunID := childWE.RunID
	p.pendingFutures[childWorkflowRunID] = future
	p.selector.AddFuture(future, func(f workflow.Future) {
		err := f.Get(ctx, nil)
		delete(p.pendingFutures, childWorkflowRunID)
		if err != nil {
			p.ChildErrors[childWorkflowRunID] = err
		}
	})

	return future, nil
}

func (p *ChildWorkflowPool) Wait() error {
	for len(p.pendingFutures) != 0 {
		p.selector.Select(p.ctx)
	}
	if len(p.ChildErrors) != 0 {
		message := ""
		for childWorkflowRunID, err := range p.ChildErrors {
			message += fmt.Sprintf("%s: %s\n", childWorkflowRunID, err)
		}
		return errors.New(message)
	}
	return nil
}
