package podalerts

import (
	"errors"
	"fmt"
	"time"

	"go.temporal.io/api/enums/v1"
	"go.temporal.io/sdk/temporal"
	"go.temporal.io/sdk/workflow"
	"gopkg.in/yaml.v2"

	ioActivities "a.yandex-team.ru/infra/temporal/activities/io"
	serviceActivities "a.yandex-team.ru/infra/temporal/activities/nanny/services"
	client "a.yandex-team.ru/infra/temporal/clients/nanny"
	servicePodAlerts "a.yandex-team.ru/infra/temporal/workflows/nanny/servicepodalerts"
)

const ConfigFileNameEnvironmentVariable = "PODALERTS_CONFIG_FILE_NAME"

type Config struct {
	TaskQueue     string   `yaml:"task_queue"`
	Namespace     string   `yaml:"namespace"`
	ServicesLimit int      `yaml:"services_limit"`
	BatchSize     int      `yaml:"batch_size"`
	Regexps       []string `yaml:"regexps"`
	MaxOffset     int      `yaml:"max_offset"`
}

type PodAlertsWorkflowState struct {
	Offset        int
	ServicesCount int
}

func CronPodAlertsWorkflow(ctx workflow.Context, config *Config, state *PodAlertsWorkflowState) error {
	cwo := workflow.ChildWorkflowOptions{
		WorkflowID:        "podAlertsCreator",
		ParentClosePolicy: enums.PARENT_CLOSE_POLICY_ABANDON,
	}
	childCtx := workflow.WithChildOptions(ctx, cwo)
	err := workflow.ExecuteChildWorkflow(childCtx, PodAlertsWorkflow, config, state).GetChildWorkflowExecution().Get(childCtx, nil)
	return err
}

func getConfig(ctx workflow.Context) (*Config, error) {
	ao := workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute,
	}
	activityCtx := workflow.WithActivityOptions(ctx, ao)

	var data []byte
	err := workflow.ExecuteActivity(
		activityCtx,
		(&ioActivities.Activities{}).GetFileContent,
		ConfigFileNameEnvironmentVariable).Get(activityCtx, &data)
	if err != nil {
		return nil, err
	}

	var config Config
	err = yaml.Unmarshal(data, &config)
	if err != nil {
		return nil, err
	}

	return &config, nil
}

func PodAlertsWorkflow(ctx workflow.Context, config *Config, state *PodAlertsWorkflowState) error {
	logger := workflow.GetLogger(ctx)
	logger.Info(fmt.Sprintf("starting pod alerts workflow with offset = %d", state.Offset))

	cfg, err := getConfig(ctx)
	if err != nil {
		return err
	}

	options := workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 10,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.0,
			MaximumAttempts:        5,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	}
	activityCtx := workflow.WithActivityOptions(ctx, options)

	var a *serviceActivities.Activities
	var services []*client.Service
	err = workflow.ExecuteActivity(
		activityCtx,
		a.GetServicesActivity,
		&serviceActivities.GetServicesRequest{
			Skip:         state.Offset,
			Limit:        cfg.BatchSize,
			Regexps:      cfg.Regexps,
			DeployEngine: "YP_LITE",
		},
	).Get(activityCtx, &services)
	if err != nil {
		return err
	}

	if cfg.ServicesLimit-state.ServicesCount < len(services) {
		services = services[:cfg.ServicesLimit-state.ServicesCount]
	}
	for _, service := range services {
		cwo := workflow.ChildWorkflowOptions{
			WorkflowID:        fmt.Sprintf("pod-alerts-%s", service.ID),
			TaskQueue:         cfg.TaskQueue,
			ParentClosePolicy: enums.PARENT_CLOSE_POLICY_ABANDON,
		}
		childCtx := workflow.WithChildOptions(ctx, cwo)
		err := workflow.ExecuteChildWorkflow(
			childCtx,
			servicePodAlerts.ServicePodAlertsWorkflow,
			&servicePodAlerts.WorkflowState{
				ServiceID:           service.ID,
				Config:              &servicePodAlerts.Config{},
				StartrekerExecution: nil,
				States:              nil,
				Info:                nil,
			}).GetChildWorkflowExecution().Get(childCtx, nil)

		if err != nil {
			// https://wiki.yandex-team.ru/awacs/development/temporal/#create-workflow-if-missed
			var cweErr *temporal.ChildWorkflowExecutionError
			if !(errors.As(err, &cweErr) && cweErr.Unwrap().Error() == "workflow execution already started") {
				return err
			}
		}
	}

	state.ServicesCount += len(services)
	state.Offset += config.BatchSize

	if state.ServicesCount >= config.ServicesLimit {
		logger.Info("reached services limit specified in config")
		return nil
	}
	if state.Offset > config.MaxOffset {
		logger.Info("workflow completed")
		return nil
	}

	return workflow.NewContinueAsNewError(ctx, PodAlertsWorkflow, config, state)
}
