package servicepodalerts

import (
	"fmt"
	"regexp"
	"time"

	"a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	"a.yandex-team.ru/infra/temporal/activities/nanny/services"
)

type PodsDecision struct {
	Faulty                    bool
	States                    []*State
	FaultyPodNames            []string
	PendingPodNames           []string
	HealthyPodNames           []string
	EvictionRequestedPodNames []string
}

type ServiceDecision struct {
	Faulty bool
	States []*State
}

type Decision struct {
	Pods    PodsDecision
	Service ServiceDecision
}

type Checker interface {
	Check([]*State, *Info) *Decision
	IsEnabled(string) (bool, error)
}

func CheckWhiteList(wl []string, serviceID string) (bool, error) {
	if len(wl) == 0 {
		return true, nil
	}

	for _, r := range wl {
		matched, err := regexp.MatchString(r, serviceID)
		if err != nil {
			return false, fmt.Errorf("regexp matching failed: %w", err)
		}
		if matched {
			return true, nil
		}
	}

	return false, nil
}

type WindowCheckerConfig struct {
	// pod is considered dead only if in the last
	// WindowSize states, in WindowThreshold of them
	// pod is not active
	WindowSize      int      `yaml:"window_size"`
	WindowThreshold int      `yaml:"window_threshold"`
	WhiteList       []string `yaml:"white_list"`
}

type PodsWindowChecker struct {
	Config *WindowCheckerConfig
}

func (wc PodsWindowChecker) IsEnabled(serviceID string) (bool, error) {
	res, err := CheckWhiteList(wc.Config.WhiteList, serviceID)
	if err != nil {
		return false, fmt.Errorf("unable to check white list for PodsWindowChecker: %w", err)
	}
	return res, nil
}

func (wc PodsWindowChecker) Check(states []*State, info *Info) *Decision {
	d := &Decision{
		Pods: PodsDecision{
			Faulty: false,
		},
	}

	var checkWindow []*State
	if len(states) > wc.Config.WindowSize {
		checkWindow = states[len(states)-wc.Config.WindowSize:]
	} else {
		checkWindow = states
	}

	faulty := make(map[string]int)
	counter := make(map[string]int)
	for _, state := range checkWindow {
		for hostName, podState := range state.PodStates {
			if _, ok := faulty[hostName]; !ok {
				faulty[hostName] = 0
				counter[hostName] = 0
			}
			if podState.State != pods.ActiveState {
				faulty[hostName] += 1
			}
			counter[hostName] += 1
		}
	}

	faultyPodNames := []string{}
	pendingPodNames := []string{}
	healthyPodNames := []string{}
	for hostName, faultCount := range faulty {
		if faultCount >= wc.Config.WindowThreshold {
			faultyPodNames = append(faultyPodNames, hostName)
		} else if wc.Config.WindowSize-counter[hostName] >= wc.Config.WindowThreshold-faultCount {
			pendingPodNames = append(pendingPodNames, hostName)
		} else {
			healthyPodNames = append(healthyPodNames, hostName)
		}
	}

	if len(faultyPodNames) != 0 {
		d.Pods.Faulty = true
	}

	d.Pods.FaultyPodNames = faultyPodNames
	d.Pods.PendingPodNames = pendingPodNames
	d.Pods.HealthyPodNames = healthyPodNames
	d.Pods.States = checkWindow
	return d
}

type ServiceWindowChecker struct {
	Config *WindowCheckerConfig
}

func (wc ServiceWindowChecker) IsEnabled(serviceID string) (bool, error) {
	res, err := CheckWhiteList(wc.Config.WhiteList, serviceID)
	if err != nil {
		return false, fmt.Errorf("unable to check white list for ServiceWindowChecker: %w", err)
	}
	return res, nil
}

func (wc ServiceWindowChecker) Check(states []*State, info *Info) *Decision {
	d := &Decision{
		Service: ServiceDecision{
			Faulty: false,
		},
	}

	var checkWindow []*State
	if len(states) > wc.Config.WindowSize {
		checkWindow = states[len(states)-wc.Config.WindowSize:]
	} else {
		checkWindow = states
	}

	faulty := 0
	for _, state := range checkWindow {
		if state.ServiceState.ServiceStatus != services.ActiveState {
			faulty++
		}
	}

	if faulty >= wc.Config.WindowThreshold {
		d.Service.Faulty = true
	}
	d.Service.States = checkWindow

	return d
}

type EvictionRequestedCheckerConfig struct {
	WhiteList                []string      `yaml:"white_list"`
	EvictionExpiredThreshold time.Duration `yaml:"eviction_expired_threshold"`
}

type EvictionRequestedChecker struct {
	Config *EvictionRequestedCheckerConfig
}

func (ec EvictionRequestedChecker) IsEnabled(serviceID string) (bool, error) {
	res, err := CheckWhiteList(ec.Config.WhiteList, serviceID)
	if err != nil {
		return false, fmt.Errorf("unable to check white list for EvictionRequestedChecker: %w", err)
	}
	return res, nil
}

func (ec EvictionRequestedChecker) Check(states []*State, info *Info) *Decision {
	d := Decision{
		Pods: PodsDecision{
			Faulty: false,
		},
	}

	annotations := info.ServiceInfo.ReplicationPolicy.GetMeta().GetAnnotations()
	notificationsEnabled := false
	if annotations != nil {
		if v, ok := annotations["notification_policy"]; ok && v == "enabled" {
			notificationsEnabled = true
		}
	}

	for podName, pod := range states[len(states)-1].PodStates {
		if !pods.IsEvictionRequested(pod.Eviction, pod.Maintenance) {
			continue
		}
		if notificationsEnabled || pod.Eviction.LastUpdated.Add(ec.Config.EvictionExpiredThreshold).Before(time.Now()) {
			d.Pods.EvictionRequestedPodNames = append(d.Pods.EvictionRequestedPodNames, podName)
		}
	}

	return &d
}

// Iteratively applies all checks and populates decision data,
// assumes states slice is consistent inside PodDecisions and ServiceDecisions
func runChecks(states []*State, checkers []Checker, info *Info, serviceID string) (*Decision, error) {
	enabledCheckers := []Checker{}
	for _, checker := range checkers {
		isEnabled, err := checker.IsEnabled(serviceID)
		if err != nil {
			return nil, err
		}
		if isEnabled {
			enabledCheckers = append(enabledCheckers, checker)
		}
	}

	decision := &Decision{}

	if len(enabledCheckers) == 0 {
		return decision, nil
	}

	decisions := []*Decision{}

	for _, checker := range enabledCheckers {
		decisions = append(decisions, checker.Check(states, info))
	}

	faulty := make(map[string]bool)
	pending := make(map[string]bool)
	healthy := make(map[string]bool)
	evictionRequested := make(map[string]bool)
	for _, d := range decisions {
		for _, podName := range d.Pods.FaultyPodNames {
			faulty[podName] = true
		}
		for _, podName := range d.Pods.PendingPodNames {
			pending[podName] = true
		}
		for _, podName := range d.Pods.HealthyPodNames {
			healthy[podName] = true
		}
		for _, podName := range d.Pods.EvictionRequestedPodNames {
			evictionRequested[podName] = true
		}
		if d.Service.Faulty {
			decision.Service.Faulty = true
		}
		// using states equality assumption
		if len(decision.Pods.States) == 0 && len(d.Pods.States) != 0 {
			decision.Pods.States = d.Pods.States
		}
		if len(decision.Service.States) == 0 && len(d.Service.States) != 0 {
			decision.Service.States = d.Service.States
		}
	}

	faultyPodNames := make([]string, 0, len(faulty))
	for podName := range faulty {
		faultyPodNames = append(faultyPodNames, podName)
		// Some decision determined this pod as faulty.
		// Even if some other decision determined it as healthy or pending
		// we have to mark it as faulty, so remove it from healthy and pending sets
		delete(healthy, podName)
		delete(pending, podName)
	}
	decision.Pods.FaultyPodNames = faultyPodNames
	if len(faulty) != 0 {
		decision.Pods.Faulty = true
	}

	pendingPodNames := make([]string, 0, len(pending))
	for podName := range pending {
		pendingPodNames = append(pendingPodNames, podName)
		delete(healthy, podName)
	}
	decision.Pods.PendingPodNames = pendingPodNames

	healthyPodNames := make([]string, 0, len(healthy))
	for podName := range healthy {
		healthyPodNames = append(healthyPodNames, podName)
	}
	decision.Pods.HealthyPodNames = healthyPodNames

	evictedPodNames := make([]string, 0, len(evictionRequested))
	for podName := range evictionRequested {
		evictedPodNames = append(evictedPodNames, podName)
	}
	decision.Pods.EvictionRequestedPodNames = evictedPodNames

	return decision, nil
}
