package servicepodalerts

import (
	"testing"
	"time"

	podsActivities "a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	"a.yandex-team.ru/infra/temporal/activities/nanny/services"
	"github.com/stretchr/testify/suite"
)

type ChecksTestSuite struct {
	suite.Suite
}

// pods: (HostName, isFaulty)
func createState(pods map[string]bool, faultyService bool) *State {
	serviceStatus := services.ActiveState
	if faultyService {
		serviceStatus = "NOTACTIVE"
	}
	state := State{
		Timestamp: time.Now(),
		ServiceState: &ServiceState{
			ServiceStatus: serviceStatus,
		},
		PodStates: make(map[string]*PodState),
	}

	for podName, isFaulty := range pods {
		podState := podsActivities.ActiveState
		if isFaulty {
			podState = "NOTACTIVE"
		}
		state.PodStates[podName] = &PodState{State: podState}
	}

	return &state
}

func (suite *ChecksTestSuite) TestPodsWindowCheckerEmpty() {
	wc := PodsWindowChecker{&WindowCheckerConfig{5, 3, []string{}}}
	d := wc.Check(nil, &Info{})

	suite.False(d.Pods.Faulty)
	suite.False(d.Service.Faulty)
}

func (suite *ChecksTestSuite) TestPodsWindowCheckerPending() {
	wc := PodsWindowChecker{&WindowCheckerConfig{5, 3, []string{}}}
	d := wc.Check(
		[]*State{
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
			}, false),
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
			}, false),
		},
		&Info{},
	)

	suite.False(d.Pods.Faulty)
	suite.ElementsMatch(d.Pods.PendingPodNames, []string{"pod1", "pod2"})
	suite.Empty(d.Pods.FaultyPodNames)
	suite.Empty(d.Pods.HealthyPodNames)
}

func (suite *ChecksTestSuite) TestPodsWindowCheckerHealthy() {
	wc := PodsWindowChecker{&WindowCheckerConfig{5, 5, []string{}}}
	d := wc.Check(
		[]*State{
			createState(map[string]bool{
				"pod1": true,
				"pod2": false,
			}, false),
			createState(map[string]bool{
				"pod1": false,
				"pod2": false,
			}, false),
		},
		&Info{},
	)

	suite.False(d.Pods.Faulty)
	suite.ElementsMatch(d.Pods.HealthyPodNames, []string{"pod1", "pod2"})
	suite.Empty(d.Pods.FaultyPodNames)
	suite.Empty(d.Pods.PendingPodNames)
}

func (suite *ChecksTestSuite) TestPodsWindowCheckerFaulty() {
	wc := PodsWindowChecker{&WindowCheckerConfig{3, 2, []string{}}}
	d := wc.Check(
		[]*State{
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
			}, false),
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
			}, false),
		},
		&Info{},
	)

	suite.True(d.Pods.Faulty)
	suite.ElementsMatch(d.Pods.FaultyPodNames, []string{"pod1", "pod2"})
	suite.Empty(d.Pods.HealthyPodNames)
	suite.Empty(d.Pods.PendingPodNames)
}

func (suite *ChecksTestSuite) TestPodsWindowCheckerMixed() {
	wc := PodsWindowChecker{&WindowCheckerConfig{5, 3, []string{}}}

	// pod1 is guaranteed healthy, it can only be fauly for 2/3 times
	// pod2 is pending, it can be faulty for 3/3 times
	// pod3 is faulty, it already has 3/3
	d := wc.Check(
		[]*State{
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
				"pod3": true,
			}, false),
			createState(map[string]bool{
				"pod1": false,
				"pod2": true,
				"pod3": true,
			}, false),
			createState(map[string]bool{
				"pod1": false,
				"pod2": false,
				"pod3": true,
			}, false),
			createState(map[string]bool{
				"pod1": false,
				"pod2": false,
				"pod3": false,
			}, false),
		},
		&Info{},
	)

	suite.True(d.Pods.Faulty)
	suite.ElementsMatch(d.Pods.FaultyPodNames, []string{"pod3"})
	suite.ElementsMatch(d.Pods.PendingPodNames, []string{"pod2"})
	suite.ElementsMatch(d.Pods.HealthyPodNames, []string{"pod1"})
}

func (suite *ChecksTestSuite) TestServiceWindowCheckerFaulty() {
	wc := ServiceWindowChecker{&WindowCheckerConfig{3, 2, []string{}}}
	d := wc.Check(
		[]*State{
			createState(map[string]bool{}, true),
			createState(map[string]bool{}, true),
		},
		&Info{},
	)

	suite.False(d.Pods.Faulty)
	suite.True(d.Service.Faulty)
}

func (suite *ChecksTestSuite) TestServiceWindowCheckerNotFaulty() {
	wc := ServiceWindowChecker{&WindowCheckerConfig{3, 2, []string{}}}
	d := wc.Check(
		[]*State{
			createState(map[string]bool{}, true),
			createState(map[string]bool{}, false),
		},
		&Info{},
	)

	suite.False(d.Pods.Faulty)
	suite.False(d.Service.Faulty)
}

func (suite *ChecksTestSuite) TestRunChecks() {
	swc := ServiceWindowChecker{&WindowCheckerConfig{5, 2, []string{}}}
	pwc := PodsWindowChecker{&WindowCheckerConfig{5, 3, []string{}}}

	d, err := runChecks(
		[]*State{
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
				"pod3": true,
			}, true),
			createState(map[string]bool{
				"pod1": false,
				"pod2": true,
				"pod3": true,
			}, false),
			createState(map[string]bool{
				"pod1": false,
				"pod2": false,
				"pod3": true,
			}, true),
			createState(map[string]bool{
				"pod1": false,
				"pod2": false,
				"pod3": false,
			}, false),
		},
		[]Checker{swc, pwc},
		&Info{},
		"test",
	)
	suite.Require().NoError(err)

	suite.True(d.Pods.Faulty)
	suite.True(d.Service.Faulty)
	suite.ElementsMatch(d.Pods.FaultyPodNames, []string{"pod3"})
	suite.ElementsMatch(d.Pods.PendingPodNames, []string{"pod2"})
	suite.ElementsMatch(d.Pods.HealthyPodNames, []string{"pod1"})
}

func (suite *ChecksTestSuite) TestRunChecksMultiplePodChecks() {
	pwc1 := PodsWindowChecker{&WindowCheckerConfig{4, 2, []string{}}}
	pwc2 := PodsWindowChecker{&WindowCheckerConfig{2, 1, []string{}}}

	// pod1 is faulty in both checks
	// pod2 is faulty in the first check, but healthy in the second
	// pod3 is pending in the first check, but healthy in the second
	// pod4 is healthy
	d, err := runChecks(
		[]*State{
			createState(map[string]bool{
				"pod2": true,
				"pod4": false,
			}, false),
			createState(map[string]bool{
				"pod1": true,
				"pod2": true,
				"pod3": true,
				"pod4": false,
			}, false),
			createState(map[string]bool{
				"pod1": true,
				"pod2": false,
				"pod3": false,
				"pod4": false,
			}, false),
			createState(map[string]bool{
				"pod1": true,
				"pod2": false,
				"pod3": false,
				"pod4": false,
			}, false),
		},
		[]Checker{pwc1, pwc2},
		&Info{},
		"test",
	)
	suite.Require().NoError(err)

	suite.True(d.Pods.Faulty)
	suite.False(d.Service.Faulty)
	suite.ElementsMatch(d.Pods.FaultyPodNames, []string{"pod1", "pod2"})
	suite.ElementsMatch(d.Pods.PendingPodNames, []string{"pod3"})
	suite.ElementsMatch(d.Pods.HealthyPodNames, []string{"pod4"})
}

func (suite *ChecksTestSuite) TestCheckWhiteList() {
	whiteList := []string{"^saas_", "test_bromigo"}

	ok, err := CheckWhiteList(whiteList, "test_bromigo")
	suite.Require().NoError(err)
	suite.True(ok)

	ok, err = CheckWhiteList(whiteList, "some_other")
	suite.Require().NoError(err)
	suite.False(ok)

	ok, err = CheckWhiteList(whiteList, "saas_test")
	suite.Require().NoError(err)
	suite.True(ok)
}

func TestChecksTestSuite(t *testing.T) {
	suite.Run(t, new(ChecksTestSuite))
}
