package servicepodalerts

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"testing"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	client "a.yandex-team.ru/infra/temporal/clients/nanny"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"gopkg.in/yaml.v2"

	"go.temporal.io/sdk/testsuite"
	"go.temporal.io/sdk/workflow"
)

var podActivities pods.Activities

type ControllerTestSuite struct {
	suite.Suite
	testsuite.WorkflowTestSuite

	env     *testsuite.TestWorkflowEnvironment
	signals map[string]interface{}
}

func (suite *ControllerTestSuite) SetupTest() {
	suite.env = suite.NewTestWorkflowEnvironment()
	// mock creating child startreker workflow
	suite.env.RegisterWorkflow(processor.ProcessWorkflow)
	suite.env.OnWorkflow(processor.ProcessWorkflow, mock.Anything, mock.Anything).
		Return(nil)
	// mock all signals, save accepted signal
	suite.env.OnSignalExternalWorkflow(mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
		Return(
			func(namespace, workflowID, runID, signalName string, arg interface{}) error {
				suite.signals[signalName] = nil
				return nil
			},
		)
}

func (suite *ControllerTestSuite) AfterTest() {
	suite.env.AssertExpectations(suite.T())
}

func (suite *ControllerTestSuite) mockGetOnDuty(logins []string) {
	suite.env.OnActivity(abcActivities.GetOnDutyFromService, mock.Anything, mock.Anything).
		Return(logins)
}

func (suite *ControllerTestSuite) mockGetServiceInfoAttrs(envType string) {
	suite.env.OnActivity(serviceActivities.GetServiceInfoAttrsActivity, mock.Anything, mock.Anything).
		Return(&client.ServiceInfoAttrs{
			EnvType: envType,
			DutySchedule: client.ServiceDutySchedule{
				ABCServiceID: 1,
				ID:           1,
			},
			AbcGroup: 1,
		}, nil)
}

func (suite *ControllerTestSuite) mockGetService(serviceStatus string, snapshotStatus pb.SnapshotStatus_Status) {
	suite.env.OnActivity(serviceActivities.GetService, mock.Anything, mock.Anything).
		Return(&pb.Service{
			Status: &pb.ServiceStatus{
				Summary: &pb.Summary{
					Value: serviceStatus,
				},
				Snapshot: []*pb.SnapshotStatus{
					{
						Status: snapshotStatus,
					},
				},
			},
		}, nil)
}

func (suite *ControllerTestSuite) mockGetServiceReplicationPolicy(rp *pb.ReplicationPolicy) {
	suite.env.OnActivity(serviceActivities.GetServiceReplicationPolicy, mock.Anything, mock.Anything).
		Return(rp, nil)
}

func (suite *ControllerTestSuite) mockGetPods(podsToReturn []*pods.Pod, expectedFaultyPods []string, totalPods int) {
	podMap := make(map[string]*pods.Pod)
	for _, pod := range podsToReturn {
		podMap[pod.PodInfo.HostName] = pod
	}
	suite.env.OnActivity(podActivities.GetPodsForServiceActivity, mock.Anything, mock.Anything, mock.Anything).
		Return(
			func(ctx context.Context, sID string, faultyPods []string) (*pods.PodsForServiceResponse, error) {
				suite.Equal(expectedFaultyPods, faultyPods)
				return &pods.PodsForServiceResponse{
					Pods:           podMap,
					TotalPodsCount: totalPods,
				}, nil
			},
		)
}

func (suite *ControllerTestSuite) executeWorkflow(state *WorkflowState) *WorkflowState {
	suite.env.ExecuteWorkflow(ServicePodAlertsWorkflow, state)
	suite.Require().True(suite.env.IsWorkflowCompleted())
	err := suite.env.GetWorkflowError()

	var canErr *workflow.ContinueAsNewError
	if !errors.As(err, &canErr) {
		suite.Require().Fail(fmt.Sprintf("unknown error type from workflow: %s", err.Error()))
	}

	payloads := canErr.Input.Payloads
	var newState *WorkflowState
	err = json.Unmarshal(payloads[0].GetData(), &newState)
	suite.Require().NoError(err)
	return newState
}

func (suite *ControllerTestSuite) getDefaultConfig() *Config {
	windowCheckerConfig := &WindowCheckerConfig{
		WindowSize:      5,
		WindowThreshold: 5,
	}

	ticketConfig := &TicketConfig{
		TaskQueue:                         "test",
		NamespaceID:                       "test",
		Queue:                             "TEST",
		Tags:                              []string{"test"},
		RetryInvocationPeriod:             time.Hour * 12,
		NannyScheduleID:                   4279,
		MaxFaultyPodsTableSize:            10,
		MaxEvictionRequestedPodsTableSize: 10,
		MaxResponsibleToSummon:            5,
		BudgetLeftThreshold:               0.,
		MinBudgetLeft:                     0,
	}

	timeoutConfig := &TimeoutConfig{
		NannyRequestTimeout: time.Second * 30,
		YpRequestTimeout:    time.Minute * 10,
		AbcRequestTimeout:   time.Minute * 2,
	}

	return &Config{
		TaskQueue:                  "test",
		PollPeriod:                 time.Hour * 4,
		StatesStoreMinLimit:        windowCheckerConfig.WindowSize,
		StatesStoreMaxLimit:        windowCheckerConfig.WindowSize * 2,
		PodsWindowCheckerConfig:    windowCheckerConfig,
		ServiceWindowCheckerConfig: windowCheckerConfig,
		EvictionRequestedCheckerConfig: &EvictionRequestedCheckerConfig{
			EvictionExpiredThreshold: time.Hour * 24,
		},
		TimeoutConfig: timeoutConfig,
		TicketConfig:  ticketConfig,
		NannyURL:      "https://nanny.yandex-team.ru",
		DeployURL:     "https://deploy.yandex-team.ru",
	}
}

func (suite *ControllerTestSuite) setConfig(config *Config) {
	out, err := yaml.Marshal(config)
	suite.Require().NoError(err)
	suite.env.OnActivity(ioActivities.GetFileContent, mock.Anything, mock.Anything).
		Return(out, nil)
}

func (suite *ControllerTestSuite) TestControllerEmptyPods() {
	suite.mockGetOnDuty([]string{"bromigo"})
	suite.mockGetServiceInfoAttrs("prod")
	suite.mockGetService("ACTIVE", pb.SnapshotStatus_ACTIVE)
	suite.mockGetServiceReplicationPolicy(
		&pb.ReplicationPolicy{
			Spec: &pb.ReplicationPolicySpec{PodGroupIdPath: "test"},
		},
	)
	suite.mockGetPods([]*pods.Pod{}, []string{}, 10)

	cfg := suite.getDefaultConfig()
	suite.setConfig(cfg)

	states := suite.executeWorkflow(&WorkflowState{
		ServiceID: "test",
	}).States

	suite.Require().Equal(1, len(states))
	suite.Equal(0, len(states[0].PodStates))
	suite.Equal(
		&ServiceState{
			SnapshotStatus: pb.SnapshotStatus_ACTIVE,
			ServiceStatus:  "ACTIVE",
		},
		states[0].ServiceState,
	)
}

func (suite *ControllerTestSuite) TestControllerNoSnapshots() {
	suite.mockGetOnDuty([]string{"bromigo"})
	suite.mockGetServiceInfoAttrs("prod")
	suite.env.OnActivity(serviceActivities.GetService, mock.Anything, mock.Anything).
		Return(&pb.Service{
			Status: &pb.ServiceStatus{
				Summary: &pb.Summary{
					Value: "ACTIVE",
				},
				Snapshot: []*pb.SnapshotStatus{},
			},
		}, nil)

	cfg := suite.getDefaultConfig()
	suite.setConfig(cfg)

	states := suite.executeWorkflow(&WorkflowState{
		ServiceID: "test",
	}).States

	suite.Equal(0, len(states))
}

func (suite *ControllerTestSuite) TestControllerHealthyPods() {
	suite.mockGetOnDuty([]string{"bromigo"})
	suite.mockGetServiceInfoAttrs("prod")
	suite.mockGetService("ACTIVE", pb.SnapshotStatus_ACTIVE)
	suite.mockGetServiceReplicationPolicy(
		&pb.ReplicationPolicy{
			Spec: &pb.ReplicationPolicySpec{PodGroupIdPath: "test"},
		},
	)
	suite.mockGetPods([]*pods.Pod{
		{
			State: "ACTIVE",
			PodInfo: &pods.PodInfo{
				PodID:    "test",
				Cluster:  "sas",
				NodeID:   "test-node-id",
				Itype:    "test",
				HostName: "test-hostname",
			},
		},
	}, []string{}, 10)

	cfg := suite.getDefaultConfig()
	suite.setConfig(cfg)

	states := suite.executeWorkflow(&WorkflowState{
		ServiceID: "test",
	}).States

	suite.Require().Equal(1, len(states))
	suite.Equal(0, len(states[0].PodStates))
}

func (suite *ControllerTestSuite) TestControllerBrokenPods() {
	suite.mockGetOnDuty([]string{"bromigo"})
	suite.mockGetServiceInfoAttrs("prod")
	suite.mockGetService("ACTIVE", pb.SnapshotStatus_ACTIVE)
	suite.mockGetServiceReplicationPolicy(
		&pb.ReplicationPolicy{
			Spec: &pb.ReplicationPolicySpec{PodGroupIdPath: "test"},
		},
	)
	suite.mockGetPods([]*pods.Pod{
		{
			State: "HOOK_SEMI_FAILED",
			PodInfo: &pods.PodInfo{
				PodID:    "test",
				Cluster:  "sas",
				NodeID:   "test-node-id",
				Itype:    "test",
				HostName: "test-hostname",
			},
			Eviction: &pods.Eviction{},
		},
	}, []string{}, 10)

	cfg := suite.getDefaultConfig()
	suite.setConfig(cfg)

	states := suite.executeWorkflow(&WorkflowState{
		ServiceID: "test",
	}).States

	suite.Require().Equal(len(states), 1)
	suite.Equal(
		map[string]*PodState{
			"test-hostname": {
				State:    "HOOK_SEMI_FAILED",
				Eviction: &pods.Eviction{},
			},
		},
		states[0].PodStates,
	)
}

func (suite *ControllerTestSuite) TestControllerPodBecameHealthy() {
	suite.mockGetOnDuty([]string{"bromigo"})
	suite.mockGetServiceInfoAttrs("prod")
	suite.mockGetService("ACTIVE", pb.SnapshotStatus_ACTIVE)
	suite.mockGetServiceReplicationPolicy(
		&pb.ReplicationPolicy{
			Spec: &pb.ReplicationPolicySpec{PodGroupIdPath: "test"},
		},
	)
	suite.mockGetPods([]*pods.Pod{
		{
			State: "ACTIVE",
			PodInfo: &pods.PodInfo{
				PodID:    "test",
				Cluster:  "sas",
				NodeID:   "test-node-id",
				Itype:    "test",
				HostName: "test-hostname",
			},
			Eviction: &pods.Eviction{},
		},
	}, []string{"test-hostname"}, 10)

	cfg := suite.getDefaultConfig()
	suite.setConfig(cfg)

	states := suite.executeWorkflow(&WorkflowState{
		ServiceID: "test",
		States: []*State{
			{
				ServiceState: &ServiceState{
					SnapshotStatus: pb.SnapshotStatus_ACTIVE,
					ServiceStatus:  "ACTIVE",
				},
				PodStates: map[string]*PodState{
					"test-hostname": {State: "HOOK_SEMI_FAILED"},
				},
			},
		},
		Info: &Info{
			PodsInfo:    map[string]*pods.PodInfo{"test-hostname": {}},
			ServiceInfo: &ServiceInfo{},
		},
	}).States

	suite.Require().Equal(2, len(states))
	suite.Equal(0, len(states[1].PodStates))
	// previous pod states must be cleared
	suite.Equal(0, len(states[0].PodStates))
}

func (suite *ControllerTestSuite) TestControllerCreateTicket() {
	suite.mockGetOnDuty([]string{"bromigo"})
	suite.mockGetServiceInfoAttrs("prod")
	suite.mockGetService("ACTIVE", pb.SnapshotStatus_ACTIVE)
	suite.mockGetServiceReplicationPolicy(
		&pb.ReplicationPolicy{
			Spec: &pb.ReplicationPolicySpec{PodGroupIdPath: "test"},
		},
	)
	suite.mockGetPods([]*pods.Pod{
		{
			State: "HOOK_SEMI_FAILED",
			PodInfo: &pods.PodInfo{
				PodID:    "test",
				Cluster:  "sas",
				NodeID:   "test-node-id",
				Itype:    "test",
				HostName: "test-hostname",
			},
			Eviction: &pods.Eviction{},
		},
	}, []string{"test-hostname"}, 10)

	cfg := suite.getDefaultConfig()
	cfg.PodsWindowCheckerConfig.WindowThreshold = 2
	suite.setConfig(cfg)

	wfState := suite.executeWorkflow(&WorkflowState{
		ServiceID: "test",
		States: []*State{
			{
				ServiceState: &ServiceState{
					SnapshotStatus: pb.SnapshotStatus_ACTIVE,
					ServiceStatus:  "ACTIVE",
				},
				PodStates: map[string]*PodState{
					"test-hostname": {State: "HOOK_SEMI_FAILED"},
				},
			},
		},
		Info: &Info{
			PodsInfo:    map[string]*pods.PodInfo{"test-hostname": {}},
			ServiceInfo: &ServiceInfo{},
		},
	})

	suite.Equal(2, len(wfState.States))
	suite.NotNil(wfState.StartrekerExecution)
}

func GetMaxUnavalablePodsWF(ctx workflow.Context, podsCount int, rp *pb.ReplicationPolicy) (int, error) {
	ctrl := PodStatesController{
		serviceID: "test",
		ctx:       ctx,
		states: []*State{
			{
				TotalPodsCount: podsCount,
			},
		},
		cfg: &Config{
			TaskQueue:     "test",
			TimeoutConfig: &TimeoutConfig{NannyRequestTimeout: time.Second},
		},
		info: &Info{
			ServiceInfo: &ServiceInfo{
				ReplicationPolicy: rp,
			},
		},
	}
	return ctrl.getMaxUnavailablePods()
}
func (suite *ControllerTestSuite) TestGetMaxUnavailablePodsSharded() {
	rp := &pb.ReplicationPolicy{
		Spec: &pb.ReplicationPolicySpec{
			PodGroupIdPath: "not empty",
		},
	}

	suite.env.ExecuteWorkflow(GetMaxUnavalablePodsWF, 10, rp)
	var maxUnavailable int
	err := suite.env.GetWorkflowResult(&maxUnavailable)
	suite.Require().NoError(err)

	suite.Equal(0, maxUnavailable)
}

func (suite *ControllerTestSuite) TestGetMaxUnavailablePodsAbsolute() {
	rp := &pb.ReplicationPolicy{
		Spec: &pb.ReplicationPolicySpec{
			DisruptionBudgetKind: pb.ReplicationPolicySpec_ABSOLUTE,
			MaxUnavailable:       5,
		},
	}

	suite.env.ExecuteWorkflow(GetMaxUnavalablePodsWF, 10, rp)
	var maxUnavailable int
	err := suite.env.GetWorkflowResult(&maxUnavailable)
	suite.Require().NoError(err)

	suite.Equal(5, maxUnavailable)
}

func (suite *ControllerTestSuite) TestGetMaxUnavailablePodsMixed() {
	rp := &pb.ReplicationPolicy{
		Spec: &pb.ReplicationPolicySpec{
			DisruptionBudgetKind: pb.ReplicationPolicySpec_MIXED,
			MaxUnavailable:       5,
		},
	}

	suite.env.ExecuteWorkflow(GetMaxUnavalablePodsWF, 10, rp)
	var maxUnavailable int
	err := suite.env.GetWorkflowResult(&maxUnavailable)
	suite.Require().NoError(err)

	suite.Equal(5, maxUnavailable)
}

func (suite *ControllerTestSuite) TestGetMaxUnavailablePodsMixedPercent() {
	rp := &pb.ReplicationPolicy{
		Spec: &pb.ReplicationPolicySpec{
			DisruptionBudgetKind: pb.ReplicationPolicySpec_MIXED,
			MaxUnavailable:       1,
		},
	}

	suite.env.ExecuteWorkflow(GetMaxUnavalablePodsWF, 100, rp)
	var maxUnavailable int
	err := suite.env.GetWorkflowResult(&maxUnavailable)
	suite.Require().NoError(err)

	// must be at least 5%, 0.05 * 100 = 5
	suite.Equal(5, maxUnavailable)
}

func (suite *ControllerTestSuite) TestGetMaxUnavailablePodsPercent() {
	rp := &pb.ReplicationPolicy{
		Spec: &pb.ReplicationPolicySpec{
			DisruptionBudgetKind:  pb.ReplicationPolicySpec_PERCENT,
			MaxUnavailablePercent: 25,
		},
	}

	suite.env.ExecuteWorkflow(GetMaxUnavalablePodsWF, 10, rp)
	var maxUnavailable int
	err := suite.env.GetWorkflowResult(&maxUnavailable)
	suite.Require().NoError(err)

	// lower(0.25 * 10) = 2
	suite.Equal(2, maxUnavailable)
}

func TestControllerTestSuite(t *testing.T) {
	suite.Run(t, new(ControllerTestSuite))
}
