package servicepodalerts

import (
	"fmt"
	"sort"
	"strings"
	"text/template"
	"time"
)

const EvictionRequestedTemplate = `{{ if .ExpiredEviction }} В вашем (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }}/ сервисе)) есть поды, эвакуация для которых запрошена более {{ .EvictionExpiredThresholdHours }} часов назад. Переселение не удалось подтвердить автоматически. Возможные причины следующие:
- Слишком много сломанных подов и не хватает бюджета на переселение
- Включено ручное подтверждение всех переселений
- Есть незавершённая выкладка (переселение работает только при отсутствии выкладки)
- В сервисе есть невыкаченный снэпшот и используется блокирующаяся об него ((https://docs.yandex-team.ru/nanny/reference/yp/replication-policy legacy-политика переселения))
{{ else }} В вашем сервисе (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }}/replication_policies включена политика уведомлений)) о подах, на которых запрошено переселение. Обычно эти нотификации включают, если переселение подов требует ручных действий, или чтобы просто быть в курсе о них. Если вам не нужны эти уведомления и ручное подтверждение переселения, вы всегда можете отключить их по ((https://wiki.yandex-team.ru/runtime-cloud/nanny/howtos/replication-policy/#notification-policy инструкции)). {{ end }}
<#
<table>
<tr>
	<td><strong>Имя пода</strong></td>
	<td><strong>Время запроса эвакуации</strong></td>
	<td><strong>Время принудительного подтверждения</strong></td>
</tr>
{{ range .EvictionRequestedPodsTable }}
<tr>
<td><a href="{{ .DeployURL }}/yp/{{ .Cluster }}/pods/{{ .PodID }}" target="_blank">{{ .HostName }}</a></td>
<td style="text-align: center;">{{ .EvictionRequestedTime }}</td>
<td style="text-align: center;">{{ .ForcedEvictionTime }}</td>
</tr>
{{ end }}
</table>
#>
{{ if gt .EvictionRequestedPodsListCount 0 }}
<{ И еще несколько подов ({{ .EvictionRequestedPodsListCount }})
	{{ .EvictionRequestedPodsList }}
}>
{{ end }}
`

type EvictionRequestedPod struct {
	DeployURL             string
	Cluster               string
	PodID                 string
	HostName              string
	EvictionRequestedTime string
	ForcedEvictionTime    string
}

type EvictionRequestedTemplateData struct {
	EvictionRequestedPodsTable     []*EvictionRequestedPod
	EvictionRequestedPodsList      string
	EvictionRequestedPodsListCount int
	ServiceID                      string
	NannyURL                       string
	EvictionExpiredThresholdHours  int

	TemplateName    string
	Name            string
	NeedsRendering  bool
	ExpiredEviction bool
}

type EvictionRequestedTemplateConfig struct {
	DeployURL                         string
	NannyURL                          string
	MaxEvictionRequestedPodsTableSize int
	ManualEvictionPeriod              time.Duration
	EvictionExpiredThreshold          time.Duration
}

type EvictionRequestedTemplateController struct {
	ServiceID string
	Decision  Decision
	Info      *Info
	Config    *EvictionRequestedTemplateConfig
	Data      *EvictionRequestedTemplateData
}

const EvictionRequestedPodsTicketTemplateName = "EvictionRequested"

var EvictionRequestedPodsTicketTemplate = template.Must(generalTemplate.New(EvictionRequestedPodsTicketTemplateName).Parse(EvictionRequestedTemplate))

func (tmpl *EvictionRequestedTemplateController) setEvictionRequestedPodsData() error {
	loc, err := time.LoadLocation("Europe/Moscow")
	if err != nil {
		return err
	}

	// sort to ensure expired evictions will go to the table
	sortedPods := make([]string, len(tmpl.Decision.Pods.EvictionRequestedPodNames))
	copy(sortedPods, tmpl.Decision.Pods.EvictionRequestedPodNames)
	states := tmpl.Decision.Pods.States[len(tmpl.Decision.Pods.States)-1].PodStates
	sort.Slice(sortedPods, func(i, j int) bool {
		ts1 := states[sortedPods[i]].Eviction.LastUpdated.Unix()
		ts2 := states[sortedPods[j]].Eviction.LastUpdated.Unix()
		return ts1 < ts2
	})

	podsList := []string{}
	for i, hostName := range sortedPods {
		pod := states[hostName]
		podInfo := tmpl.Info.PodsInfo[hostName]

		if i >= tmpl.Config.MaxEvictionRequestedPodsTableSize {
			podsList = append(podsList, podInfo.PodID)
			continue
		}

		evictionRequestedTS := pod.Eviction.LastUpdated.In(loc)
		evictionForcedTS := evictionRequestedTS.Add(tmpl.Config.ManualEvictionPeriod)
		tmpl.Data.EvictionRequestedPodsTable = append(tmpl.Data.EvictionRequestedPodsTable, &EvictionRequestedPod{
			HostName:              hostName,
			Cluster:               podInfo.Cluster,
			PodID:                 podInfo.PodID,
			DeployURL:             tmpl.Config.DeployURL,
			EvictionRequestedTime: evictionRequestedTS.Format("02 Jan 15:04 MST"),
			ForcedEvictionTime:    evictionForcedTS.Format("02 Jan 15:04 MST"),
		})

		if pod.Eviction.LastUpdated.Add(tmpl.Config.EvictionExpiredThreshold).Before(time.Now()) {
			tmpl.Data.ExpiredEviction = true
		}
	}

	tmpl.Data.EvictionRequestedPodsList = strings.Join(podsList, ", ")
	tmpl.Data.EvictionRequestedPodsListCount = len(podsList)

	return nil
}

func (tmpl *EvictionRequestedTemplateController) Init() error {
	if len(tmpl.Decision.Pods.EvictionRequestedPodNames) == 0 {
		tmpl.Data = &EvictionRequestedTemplateData{
			NeedsRendering: false,
		}
		return nil
	}

	tmpl.Data = &EvictionRequestedTemplateData{
		NannyURL:                      tmpl.Config.NannyURL,
		ServiceID:                     tmpl.ServiceID,
		NeedsRendering:                true,
		Name:                          "Поды с запрошенной эвакуацией",
		TemplateName:                  EvictionRequestedPodsTicketTemplateName,
		EvictionExpiredThresholdHours: int(tmpl.Config.EvictionExpiredThreshold.Hours()),
	}

	return tmpl.setEvictionRequestedPodsData()
}

func (tmpl *EvictionRequestedTemplateController) GetInvocationText() string {
	if !tmpl.Data.NeedsRendering {
		return ""
	}

	text := fmt.Sprintf(
		"В сервисе обнаружено %d подов с запрошенной эвакуацией. ",
		len(tmpl.Data.EvictionRequestedPodsList)+len(tmpl.Data.EvictionRequestedPodsTable),
	)
	if tmpl.Data.ExpiredEviction {
		text += fmt.Sprintf("Среди них есть те, у которых эвакуация запрошена больше %d часов назад. ",
			tmpl.Data.EvictionExpiredThresholdHours)
	} else {
		text += fmt.Sprintf("У вас ((%s/ui/#/services/catalog/%s/replication_policies включены уведомления)) о них. ",
			tmpl.Config.NannyURL,
			tmpl.ServiceID,
		)
	}

	return text
}
