package servicepodalerts

import (
	"bytes"
	"testing"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

type EvictionRequestedTemplateTestSuite struct {
	suite.Suite
}

func (suite *EvictionRequestedTemplateTestSuite) TestTemplateExecution() {
	cfg := &EvictionRequestedTemplateConfig{
		NannyURL:                          "https://nanny.yandex-team.ru",
		DeployURL:                         "https://deploy.yandex-team.ru",
		MaxEvictionRequestedPodsTableSize: 10,
		EvictionExpiredThreshold:          time.Hour * 24,
	}

	info := &Info{
		PodsInfo: map[string]*pods.PodInfo{
			"some-pod-hostname": {
				PodID:    "some-pod",
				Cluster:  "sas",
				NodeID:   "some-node-id",
				Itype:    "default",
				HostName: "some-pod-hostname",
			},
		},
		ServiceInfo: &ServiceInfo{},
	}

	states := []*State{
		{
			Timestamp: time.Now(),
			ServiceState: &ServiceState{
				SnapshotStatus: pb.SnapshotStatus_ACTIVE,
				ServiceStatus:  "ACTIVE",
			},
			PodStates: map[string]*PodState{
				"some-pod-hostname": {
					State: "HOOK_SEMI_FAILED",
					Eviction: &pods.Eviction{
						LastUpdated: time.Now(),
					},
				},
			},
		},
	}

	decision := Decision{
		Pods: PodsDecision{
			Faulty:                    true,
			States:                    states,
			EvictionRequestedPodNames: []string{"some-pod-hostname"},
		},
		Service: ServiceDecision{
			States: states,
		},
	}

	tmpl := &EvictionRequestedTemplateController{
		ServiceID: "test",
		Decision:  decision,
		Info:      info,
		Config:    cfg,
	}

	err := tmpl.Init()
	require.NoError(suite.T(), err)

	buf := new(bytes.Buffer)
	err = EvictionRequestedPodsTicketTemplate.Execute(buf, tmpl.Data)
	require.NoError(suite.T(), err)
}

func (suite *EvictionRequestedTemplateTestSuite) TestEvictionRequestedPodsData() {
	tmpl := &EvictionRequestedTemplateController{
		Decision: Decision{
			Pods: PodsDecision{
				EvictionRequestedPodNames: []string{"hostname"},
				States: []*State{
					{
						PodStates: map[string]*PodState{
							"hostname": {
								Eviction: &pods.Eviction{
									LastUpdated: time.Unix(0, 0),
								},
							},
						},
					},
				},
			},
		},
		Config: &EvictionRequestedTemplateConfig{
			DeployURL:                         "deploy",
			MaxEvictionRequestedPodsTableSize: 2,
			ManualEvictionPeriod:              time.Hour * 24 * 4,
			EvictionExpiredThreshold:          time.Hour * 24,
		},
		Info: &Info{
			PodsInfo: map[string]*pods.PodInfo{
				"hostname": {
					PodID:   "podid",
					Cluster: "cluster",
				},
			},
		},
	}
	tmpl.Data = &EvictionRequestedTemplateData{}
	err := tmpl.setEvictionRequestedPodsData()
	require.NoError(suite.T(), err)

	require.Equal(suite.T(), 0, tmpl.Data.EvictionRequestedPodsListCount)
	require.Equal(
		suite.T(),
		[]*EvictionRequestedPod{
			{
				DeployURL:             "deploy",
				Cluster:               "cluster",
				PodID:                 "podid",
				HostName:              "hostname",
				EvictionRequestedTime: "01 Jan 03:00 MSK",
				ForcedEvictionTime:    "05 Jan 03:00 MSK",
			},
		},
		tmpl.Data.EvictionRequestedPodsTable,
	)
}

func (suite *EvictionRequestedTemplateTestSuite) TestEvictionRequestedPodsList() {
	tmpl := &EvictionRequestedTemplateController{
		Decision: Decision{
			Pods: PodsDecision{
				EvictionRequestedPodNames: []string{"pod1", "pod2", "pod3"},
				States: []*State{
					{
						PodStates: map[string]*PodState{
							"pod1": {Eviction: &pods.Eviction{LastUpdated: time.Unix(2, 0)}},
							"pod2": {Eviction: &pods.Eviction{LastUpdated: time.Unix(1, 0)}},
							"pod3": {Eviction: &pods.Eviction{LastUpdated: time.Unix(0, 0)}},
						},
					},
				},
			},
		},
		Config: &EvictionRequestedTemplateConfig{
			MaxEvictionRequestedPodsTableSize: 1,
			EvictionExpiredThreshold:          time.Hour * 24,
		},
		Info: &Info{
			PodsInfo: map[string]*pods.PodInfo{"pod1": {PodID: "pod1"}, "pod2": {PodID: "pod2"}, "pod3": {PodID: "pod3"}},
		},
	}
	tmpl.Data = &EvictionRequestedTemplateData{}
	err := tmpl.setEvictionRequestedPodsData()
	require.NoError(suite.T(), err)

	require.Equal(suite.T(), 1, len(tmpl.Data.EvictionRequestedPodsTable))
	require.Equal(suite.T(), 2, tmpl.Data.EvictionRequestedPodsListCount)
	// pod3 is the oldest, so it goes to table
	require.Equal(suite.T(), "pod2, pod1", tmpl.Data.EvictionRequestedPodsList)
	require.Equal(suite.T(), "pod3", tmpl.Data.EvictionRequestedPodsTable[0].HostName)
}

func TestEvictionRequestedTemplateTestSuite(t *testing.T) {
	suite.Run(t, new(EvictionRequestedTemplateTestSuite))
}
