package servicepodalerts

import (
	"bytes"
	"strings"
	"text/template"
	"time"

	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
)

type TemplateController interface {
	Init() error
	GetInvocationText() string
}

const GeneralTemplate = `====В сервисе обнаружены:
{{ range $idx, $tmpl := .NotificationTemplates }}
	<a href="#{{ .Data.TemplateName }}">{{- inc $idx }}. {{ .Data.Name }}</a>
{{ end }}
Детали приведены ниже
{{ range .NotificationTemplates }}
	{{- /* You can't have variable nested template names */ -}}
	===<p id="{{ .Data.TemplateName }}">{{- .Data.Name}}<p>
	{{ if eq .Data.TemplateName "BrokenPods" }}
		{{ template "BrokenPods" .Data }}
	{{ else if eq .Data.TemplateName "EvictionRequested" }}
		{{- template "EvictionRequested" .Data }}
	{{ end }}
{{ end }}

==== Мне не нравятся эти тикеты
Мы начали внедрять инфраструктуру подобных уведомлений не так давно, она может быть несовершенна и мы с радостью её улучшим. Если кажется, что наши критерии сломанности подов слишком жесткие, нотификации бесят, инструкция непонятная или что-то ещё плохо - оставьте, пожалуйста, конструктивный наброс комментом к этому тикету и призовите дежурного, нажав на кнопку "Призвать дежурного"`

var generalTemplateFuncMap = template.FuncMap{
	"inc": func(i int) int {
		return i + 1
	},
}

var generalTemplate = template.Must(template.New("General").Funcs(generalTemplateFuncMap).Parse(GeneralTemplate))

type GeneralTemplateConfig struct {
	NannyScheduleID                   int
	NannyURL                          string
	DeployURL                         string
	PollPeriod                        time.Duration
	MaxFaultyPodsTableSize            int
	MaxEvictionRequestedPodsTableSize int
	ManualEvictionPeriod              time.Duration
	EvictionExpiredThreshold          time.Duration
}

type GeneralTemplateController struct {
	ServiceID               string
	Decision                Decision
	Info                    *Info
	Config                  *GeneralTemplateConfig
	RetryInvocationSettings *processor.RetryInvocationSettings
	Data                    *GeneralTemplateData
}

type GeneralTemplateData struct {
	NotificationTemplates []TemplateController
}

func (tmpl *GeneralTemplateController) Init() error {
	tmpl.Data = &GeneralTemplateData{}

	podsAlertTemplateController := &PodsAlertTemplateController{
		ServiceID:               tmpl.ServiceID,
		Decision:                tmpl.Decision,
		Info:                    tmpl.Info,
		RetryInvocationSettings: tmpl.RetryInvocationSettings,
		Config: &PodsAlertTemplateConfig{
			NannyURL:               tmpl.Config.NannyURL,
			DeployURL:              tmpl.Config.DeployURL,
			PollPeriod:             tmpl.Config.PollPeriod,
			MaxFaultyPodsTableSize: tmpl.Config.MaxFaultyPodsTableSize,
		},
	}
	err := podsAlertTemplateController.Init()
	if err != nil {
		return err
	}
	if podsAlertTemplateController.Data.NeedsRendering {
		tmpl.Data.NotificationTemplates = append(tmpl.Data.NotificationTemplates, podsAlertTemplateController)
	}

	evictionRequestedTemplateController := &EvictionRequestedTemplateController{
		ServiceID: tmpl.ServiceID,
		Decision:  tmpl.Decision,
		Info:      tmpl.Info,
		Config: &EvictionRequestedTemplateConfig{
			DeployURL:                         tmpl.Config.DeployURL,
			NannyURL:                          tmpl.Config.NannyURL,
			MaxEvictionRequestedPodsTableSize: tmpl.Config.MaxEvictionRequestedPodsTableSize,
			ManualEvictionPeriod:              tmpl.Config.ManualEvictionPeriod,
			EvictionExpiredThreshold:          tmpl.Config.EvictionExpiredThreshold,
		},
	}
	err = evictionRequestedTemplateController.Init()
	if err != nil {
		return err
	}
	if evictionRequestedTemplateController.Data.NeedsRendering {
		tmpl.Data.NotificationTemplates = append(tmpl.Data.NotificationTemplates, evictionRequestedTemplateController)
	}

	return nil
}

func (tmpl *GeneralTemplateController) GetInvocationText() string {
	var invocationTexts []string
	for _, tmpl := range tmpl.Data.NotificationTemplates {
		invocationTexts = append(invocationTexts, tmpl.GetInvocationText())
	}
	return strings.Join(invocationTexts, "\n\n")
}

func (tmpl *GeneralTemplateController) Render() (string, error) {
	buf := new(bytes.Buffer)
	err := generalTemplate.Execute(buf, tmpl.Data)
	if err != nil {
		return "", err
	}
	return buf.String(), nil
}
