package servicepodalerts

import (
	"fmt"
	"strings"
	"text/template"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
)

const PodsTicketTemplate = `В сервисе (({{ .NannyURL }}/ui/#/services/catalog/{{.ServiceID}} {{.ServiceID}})) обнаружены сломанные поды. Такое случается, если:
* падает сам запускаемый в контейнере сервис
* приложение в контейнере запущено, но сломалась его внутренняя логика и теперь не проходит ((https://docs.yandex-team.ru/nanny/how-to/structured-instancectl-config#readiness проверка его статуса)).
* на хосте возникли проблемы с железом или хостовой инфраструктурой, и контейнеру стало плохо.

==== История состояний сервиса
<#
<table>
<tr>
<td></td>
{{ range $index, $ts := .Timestamps }}
<td style="text-align: center;">{{ $ts }}</td>
{{ end }}
</tr>
<tr>
<td>Статус снапшота</td>
{{ range .SnapshotStatuses }}
<td><div title="{{ .Text }}" style="width: 160px; display: inline-block; text-align: center; background: {{ .Color }}; color: white; ">{{ .Text }}</div></td>
{{ end }}
</tr>

<tr>
<td><p style="font-size: large;">Проблемные поды</p></td>
</tr>

{{ range .FaultyPodsTable }}
<tr>
<td><a href="{{ .DeployURL }}/yp/{{ .Cluster }}/pods/{{ .PodID }}" target="_blank">{{ .HostName }}</a><br>
Графики: <a href="https://yasm.yandex-team.ru/template/panel/New-Porto-Container/hosts={{ .HostName }};itype={{ .Itype }};nanny={{ .ServiceID }};node={{ .NodeID }}" target="_blank">под</a>,
<a href="https://yasm.yandex-team.ru/template/panel/Host/hosts={{ .NodeID }}" target="_blank">хост</a></td>
{{ range .States }}
<td><div title="{{ .Text }}" style=" width: 160px; display: inline-block; text-align: center; background: {{ .Color }}; color: white; ">{{ .Text }}</div></td>
{{ end }}
</tr>
{{ end }}
</table>
{{ if gt .FaultyPodsListCount 0 }}
	#>

	<{ и еще несколько подов ({{ .FaultyPodsListCount }})
	{{ .FaultyPodsList }}
	}>
	<#
{{ end }}
#>

((https://wiki.yandex-team.ru/runtime-cloud/nanny/howto-manage-service/#iss-states-definitions Легенда статусов))

==== Почему это важно?
Это может значить, что сервис сломан целиком. Но даже если это не так, мы всё равно рекомендуем починить сломанные поды.
<{Если долго не делать этого, то на других подах сервиса могут начаться регламентные работы с даунтаймом.
Все регламентные работы в облаке запускаются с учётом бюджета на ((https://wiki.yandex-team.ru/runtime-cloud/nanny/howtos/replication-policy/ переселение подов)), указываемого в %%Replication policy%% сервиса. В данный бюджет входят также все сломанные поды. Иными словами, если бюджет сервиса на переселение равен 10 подов, и в сервисе уже сломано 10 подов по каким-то случайным причинам, то ни один другой под сервиса не будет отдан в даунтайм для проведения регламентных работ. Однако есть и исключение, связанное с тем, что **таймаут ожидания запуска работ ограничен**. Поэтому если текущие сломанные поды в течение длительного времени не оживут, работы на других подах сервиса всё равно будут запущены. Это приведёт к даунтайму других подов сервиса, и суммарно в нерабочем состоянии окажется уже большее число контейнеров сервиса, и оно уже может быть критичным.}>

==== Как починить поломку?
Нужно разобраться в её причинах, для чего провести диагностику:
1. Открыть (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }} сервис)) и почитать его (({{ .NannyURL }}/ui/#/services/catalog/{{ .ServiceID }}/runtime_attrs_history историю выкладок)), возможно в нём катилось что-то ломающее.
2. Посмотреть текущее состояние сломанных подов, например, (({{ .NannyURL }}/ui/#/services/pods/{{ .ServiceID }}/clusters/{{ .PodExample.Cluster | GetNannyClusterName }}/pods/{{ .PodExample.PodID }} {{ .PodExample.HostName }})) и сверить его с ((https://wiki.yandex-team.ru/runtime-cloud/nanny/howto-manage-service/#iss-states-definitions легендой статусов)).
3. Зайти в контейнер пода через команду %%ssh {{ .PodExample.HostName }}%% и почитать логи инфраструктурных супервизоров: %%loop.log%%, %%loop.log.full%%, %%anamnesis.log%%. Если какой-то из процессов сервиса падает, то зачастую в этих логах можно узнать, какой именно.
4. Почитать stderr и stdout процессов сервиса, для этого как и выше зайти в контейнер пода и посмотреть файлы с расширениями %%.err%% и %%.out%%
5. Помимо логов инфраструктурных процессов есть смысл почитать логи самого запущенного сервиса, как правило они находятся в директории %%/logs%%, но могут и в другом месте, это зависит от того, как свой сервис настраивал его владелец.
6. Посмотреть графики потребления ресурсов сломанными подами, ((https://yasm.yandex-team.ru/template/panel/New-Porto-Container/hosts={{ .PodExample.HostName }};itype={{ .PodExample.Itype}};nanny={{ .ServiceID }};node={{ .PodExample.NodeID }} пример)). Возможно сервису перестало хватать вычислительных ресурсов, например, памяти или процессора.
7. Если по логам и графикам ничего непонятно, починить надо, но не получается, то призвать дежурного, нажав на кнопку "Призвать дежурного". Спустя какое-то время дежурный от команды Nanny поможет с диагностикой проблем.

Если сервис //штатно сломан//, например, это тестовый или временно поднятый контур, то чтобы подобные тикеты не заводились, достаточно ((https://wiki.yandex-team.ru/runtime-cloud/nanny/howto-manage-service/#mark-as-testing пометить его как testing)). Но делать так с продовыми сервисами не рекомендуется, т.к. это значительно ослабит ограничения на даунтайм их подов для работ на железе.

==== Я починил под, когда закроется тикет?
Статусы подов пересчитываются раз в несколько часов (на момент последнего обновления тикета, число часов - {{ .PollPeriodHours }}). Как только сломанных подов не останется мы обязательно пришлем уведомление и закроем тикет.`

type Color string

const SuccesColor Color = "#5cb85c"
const PrimaryColor Color = "#337ab7"
const InfoColor Color = "#5bc0de"
const WarningColor Color = "#f0ad4e"
const DangerColor Color = "#dc3545"
const DefaultColor Color = "#777777"

type SnapshotStatusTemplateData struct {
	Text  string
	Color Color
}

type PodStateTemplateData struct {
	Text  string
	Color Color
}

type PodTemplateData struct {
	ServiceID string
	DeployURL string
	Cluster   string
	PodID     string
	HostName  string
	Itype     string
	NodeID    string

	States []*PodStateTemplateData
}

type PodsAlertTemplateData struct {
	ServiceID           string
	DeployURL           string
	NannyURL            string
	Timestamps          []string
	SnapshotStatuses    []*SnapshotStatusTemplateData
	FaultyPodsTable     []*PodTemplateData
	FaultyPodsList      string
	FaultyPodsListCount int
	PodExample          *PodTemplateData
	PollPeriodHours     int

	Name           string
	TemplateName   string
	NeedsRendering bool
}

type PodsAlertTemplateConfig struct {
	NannyURL               string
	DeployURL              string
	PollPeriod             time.Duration
	MaxFaultyPodsTableSize int
}

type PodsAlertTemplateController struct {
	ServiceID               string
	Decision                Decision
	Info                    *Info
	Config                  *PodsAlertTemplateConfig
	Data                    *PodsAlertTemplateData
	RetryInvocationSettings *processor.RetryInvocationSettings
}

func GetNannyClusterName(cluster string) string {
	switch cluster {
	case "sas-test":
		return "TEST_SAS"
	case "man-pre":
		return "MAN_PRE"
	default:
		return strings.ToUpper(cluster)
	}
}

var podsAlertTemplateFuncMap = template.FuncMap{
	"GetNannyClusterName": GetNannyClusterName,
}

const PodsTicketTemplateName = "BrokenPods"

var podsTicketTemplate = template.Must(generalTemplate.New(PodsTicketTemplateName).Funcs(podsAlertTemplateFuncMap).Parse(PodsTicketTemplate))

func (tmpl *PodsAlertTemplateController) setTimestamps() error {
	loc, err := time.LoadLocation("Europe/Moscow")
	if err != nil {
		return err
	}
	for _, state := range tmpl.Decision.Service.States {
		tmpl.Data.Timestamps = append(tmpl.Data.Timestamps, state.Timestamp.In(loc).Format("02 Jan 15:04 MST"))
	}
	return nil
}

func (tmpl *PodsAlertTemplateController) setSnapshotsData() {
	for _, state := range tmpl.Decision.Service.States {
		snapshotSatus := state.ServiceState.SnapshotStatus

		var color Color
		switch snapshotSatus {
		case pb.SnapshotStatus_PREPARED:
			color = SuccesColor
		case pb.SnapshotStatus_ACTIVE:
			color = PrimaryColor
		case pb.SnapshotStatus_PREPARING, pb.SnapshotStatus_ACTIVATING, pb.SnapshotStatus_GENERATING:
			color = InfoColor
		case pb.SnapshotStatus_DEACTIVATE_PENDING:
			color = WarningColor
		default:
			color = DefaultColor
		}

		tmpl.Data.SnapshotStatuses = append(tmpl.Data.SnapshotStatuses, &SnapshotStatusTemplateData{
			Text:  pb.SnapshotStatus_Status_name[int32(state.ServiceState.SnapshotStatus)],
			Color: color,
		})
	}
}

func (tmpl *PodsAlertTemplateController) setPodsData() {
	faultyPodsList := []string{}

	for i, podName := range tmpl.Decision.Pods.FaultyPodNames {
		podInfo := tmpl.Info.PodsInfo[podName]

		if i >= tmpl.Config.MaxFaultyPodsTableSize {
			faultyPodsList = append(faultyPodsList, podInfo.PodID)
			continue
		}

		podTemplateData := &PodTemplateData{
			ServiceID: tmpl.ServiceID,
			DeployURL: tmpl.Config.DeployURL,
			Cluster:   podInfo.Cluster,
			PodID:     podInfo.PodID,
			HostName:  podInfo.HostName,
			Itype:     podInfo.Itype,
			NodeID:    podInfo.NodeID,
		}

		for _, state := range tmpl.Decision.Pods.States {
			podState := state.PodStates[podName].State
			var color Color
			switch strings.ToLower(podState) {
			case "active":
				color = PrimaryColor
			case "prepared":
				color = SuccesColor
			case "removed":
				color = WarningColor
			case "error":
				color = DangerColor
			default:
				color = DefaultColor
			}

			podTemplateData.States = append(podTemplateData.States, &PodStateTemplateData{
				Text:  state.PodStates[podName].State,
				Color: color,
			})
		}

		tmpl.Data.FaultyPodsTable = append(tmpl.Data.FaultyPodsTable, podTemplateData)
	}

	if len(tmpl.Data.FaultyPodsTable) > 0 {
		tmpl.Data.PodExample = tmpl.Data.FaultyPodsTable[0]
	}
	tmpl.Data.FaultyPodsList = strings.Join(faultyPodsList, ", ")
	tmpl.Data.FaultyPodsListCount = len(faultyPodsList)
}

func (tmpl *PodsAlertTemplateController) Init() error {
	if len(tmpl.Decision.Pods.FaultyPodNames) == 0 {
		tmpl.Data = &PodsAlertTemplateData{
			NeedsRendering: false,
		}
		return nil
	}

	tmpl.Data = &PodsAlertTemplateData{
		ServiceID:       tmpl.ServiceID,
		DeployURL:       tmpl.Config.DeployURL,
		NannyURL:        tmpl.Config.NannyURL,
		PollPeriodHours: int(tmpl.Config.PollPeriod.Hours()),
		NeedsRendering:  true,
		Name:            "Сломанные поды",
		TemplateName:    PodsTicketTemplateName,
	}

	err := tmpl.setTimestamps()
	if err != nil {
		return err
	}
	tmpl.setSnapshotsData()
	tmpl.setPodsData()
	return nil
}

func (tmpl *PodsAlertTemplateController) GetInvocationText() string {
	if !tmpl.Data.NeedsRendering {
		return ""
	}

	var text string
	if tmpl.Info.ServiceInfo.MaxUnavailablePods != 0 {
		text = fmt.Sprintf(
			"Недоступно %d подов при бюджете в %d. ",
			len(tmpl.Decision.Pods.FaultyPodNames),
			tmpl.Info.ServiceInfo.MaxUnavailablePods,
		)
	} else {
		text = fmt.Sprintf(
			"Недоступно %d подов для шардированного сервиса. ",
			len(tmpl.Decision.Pods.FaultyPodNames),
		)
	}

	switch tmpl.RetryInvocationSettings.Kind {
	case processor.Once:
		text += "Это находится в допустимых пределах, как только количество мертвых подов станет критичным, мы призовем ответственного в этот тикет. "
	case processor.Period:
		text += "Регламентные работы на хостах сервиса могут вызвать превышение бюджета недоступных подов. Это может привести к поломке сервиса. "
	}

	return text
}
