package servicepodalerts

import (
	"bytes"
	"testing"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

type PodsTemplateTestSuite struct {
	suite.Suite
}

func (suite *PodsTemplateTestSuite) TestTemplateInit() {
	cfg := &PodsAlertTemplateConfig{
		NannyURL:               "https://nanny.yandex-team.ru",
		DeployURL:              "https://deploy.yandex-team.ru",
		PollPeriod:             time.Hour * 4,
		MaxFaultyPodsTableSize: 10,
	}

	info := &Info{
		PodsInfo: map[string]*pods.PodInfo{
			"some-pod-hostname": {
				PodID:    "some-pod",
				Cluster:  "sas",
				NodeID:   "some-node-id",
				Itype:    "default",
				HostName: "some-pod-hostname",
			},
		},
		ServiceInfo: &ServiceInfo{},
	}

	states := []*State{
		{
			Timestamp: time.Now(),
			ServiceState: &ServiceState{
				SnapshotStatus: pb.SnapshotStatus_ACTIVE,
				ServiceStatus:  "ACTIVE",
			},
			PodStates: map[string]*PodState{
				"some-pod-hostname": {
					State: "HOOK_SEMI_FAILED",
				},
			},
		},
	}

	decision := Decision{
		Pods: PodsDecision{
			Faulty:         true,
			States:         states,
			FaultyPodNames: []string{"some-pod-hostname"},
		},
		Service: ServiceDecision{
			States: states,
		},
	}

	tmpl := &PodsAlertTemplateController{
		ServiceID:               "test",
		Decision:                decision,
		Info:                    info,
		Config:                  cfg,
		RetryInvocationSettings: &processor.RetryInvocationSettings{Kind: processor.Once},
	}

	err := tmpl.Init()
	require.NoError(suite.T(), err)

	buf := new(bytes.Buffer)
	err = podsTicketTemplate.Execute(buf, tmpl.Data)
	require.NoError(suite.T(), err)
}

func (suite *PodsTemplateTestSuite) TestGetNannyClusterName() {
	require.Equal(suite.T(), "TEST_SAS", GetNannyClusterName("sas-test"))
	require.Equal(suite.T(), "MAN_PRE", GetNannyClusterName("man-pre"))
	require.Equal(suite.T(), "MAN", GetNannyClusterName("man"))
	require.Equal(suite.T(), "SAS", GetNannyClusterName("SAS"))
}

func (suite *PodsTemplateTestSuite) TestSetTimestamps() {
	tmpl := &PodsAlertTemplateController{
		Decision: Decision{
			Service: ServiceDecision{
				States: []*State{
					{Timestamp: time.Unix(0, 0)},
					{Timestamp: time.Unix(60, 0)},
				},
			},
		},
	}
	tmpl.Data = &PodsAlertTemplateData{}
	err := tmpl.setTimestamps()
	require.NoError(suite.T(), err)

	require.Equal(
		suite.T(),
		[]string{"01 Jan 03:00 MSK", "01 Jan 03:01 MSK"},
		tmpl.Data.Timestamps,
	)
}

func (suite *PodsTemplateTestSuite) TestSetSnapshotsData() {
	tmpl := &PodsAlertTemplateController{
		Decision: Decision{
			Service: ServiceDecision{
				States: []*State{
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_BROKEN}},
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_PREPARED}},
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_ACTIVE}},
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_PREPARING}},
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_ACTIVATING}},
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_GENERATING}},
					{ServiceState: &ServiceState{SnapshotStatus: pb.SnapshotStatus_DEACTIVATE_PENDING}},
				},
			},
		},
	}
	tmpl.Data = &PodsAlertTemplateData{}
	tmpl.setSnapshotsData()

	require.Equal(
		suite.T(),
		[]*SnapshotStatusTemplateData{
			{Text: "BROKEN", Color: DefaultColor},
			{Text: "PREPARED", Color: SuccesColor},
			{Text: "ACTIVE", Color: PrimaryColor},
			{Text: "PREPARING", Color: InfoColor},
			{Text: "ACTIVATING", Color: InfoColor},
			{Text: "GENERATING", Color: InfoColor},
			{Text: "DEACTIVATE_PENDING", Color: WarningColor},
		},
		tmpl.Data.SnapshotStatuses,
	)
}

func (suite *PodsTemplateTestSuite) TestSetPodsData() {
	tmpl := &PodsAlertTemplateController{
		Decision: Decision{
			Pods: PodsDecision{
				FaultyPodNames: []string{"hostname"},
				States: []*State{
					{PodStates: map[string]*PodState{"hostname": {State: "ACTIVE"}}},
					{PodStates: map[string]*PodState{"hostname": {State: "PREPARED"}}},
					{PodStates: map[string]*PodState{"hostname": {State: "REMOVED"}}},
					{PodStates: map[string]*PodState{"hostname": {State: "ERROR"}}},
					{PodStates: map[string]*PodState{"hostname": {State: "DEFAULT"}}},
				},
			},
		},
		ServiceID: "test",
		Config: &PodsAlertTemplateConfig{
			DeployURL:              "deploy",
			MaxFaultyPodsTableSize: 10,
		},
		Info: &Info{
			PodsInfo: map[string]*pods.PodInfo{
				"hostname": {
					Cluster:  "cluster",
					PodID:    "podid",
					HostName: "hostname",
					Itype:    "itype",
					NodeID:   "nodeid",
				},
			},
		},
	}
	tmpl.Data = &PodsAlertTemplateData{}
	tmpl.setPodsData()

	require.Equal(suite.T(), 0, tmpl.Data.FaultyPodsListCount)
	require.Equal(
		suite.T(),
		[]*PodTemplateData{
			{
				ServiceID: "test",
				DeployURL: "deploy",
				Cluster:   "cluster",
				PodID:     "podid",
				HostName:  "hostname",
				Itype:     "itype",
				NodeID:    "nodeid",
				States: []*PodStateTemplateData{
					{Text: "ACTIVE", Color: PrimaryColor},
					{Text: "PREPARED", Color: SuccesColor},
					{Text: "REMOVED", Color: WarningColor},
					{Text: "ERROR", Color: DangerColor},
					{Text: "DEFAULT", Color: DefaultColor},
				},
			},
		},
		tmpl.Data.FaultyPodsTable,
	)
	require.Equal(suite.T(), "hostname", tmpl.Data.PodExample.HostName)
}

func (suite *PodsTemplateTestSuite) TestSetPodsDataFaultyPodsList() {
	tmpl := &PodsAlertTemplateController{
		Decision: Decision{
			Pods: PodsDecision{
				FaultyPodNames: []string{"pod1", "pod2", "pod3"},
				States: []*State{
					{PodStates: map[string]*PodState{"pod1": {State: "ERROR"}, "pod2": {State: "ERROR"}, "pod3": {State: "ERROR"}}},
				},
			},
		},
		Config: &PodsAlertTemplateConfig{
			MaxFaultyPodsTableSize: 1,
		},
		Info: &Info{
			PodsInfo: map[string]*pods.PodInfo{"pod1": {PodID: "pod1"}, "pod2": {PodID: "pod2"}, "pod3": {PodID: "pod3"}},
		},
	}
	tmpl.Data = &PodsAlertTemplateData{}
	tmpl.setPodsData()

	require.Equal(suite.T(), 1, len(tmpl.Data.FaultyPodsTable))
	require.Equal(suite.T(), 2, tmpl.Data.FaultyPodsListCount)
	require.Equal(suite.T(), "pod2, pod3", tmpl.Data.FaultyPodsList)
}

func TestPodsTemplateTestSuite(t *testing.T) {
	suite.Run(t, new(PodsTemplateTestSuite))
}
