package main

import (
	"fmt"
	"log"
	"os"
	"time"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	"a.yandex-team.ru/infra/temporal/workflows/nanny/servicepodalerts"
)

const OutFile = "podstemplate.md"

func generateStates(podNames []string, podStates []string, serviceStates []pb.SnapshotStatus_Status) []*servicepodalerts.State {
	states := []*servicepodalerts.State{}
	for i, state := range podStates {
		podStates := make(map[string]*servicepodalerts.PodState)
		for _, pod := range podNames {
			podStates[pod] = &servicepodalerts.PodState{
				State: state,
				Eviction: &pods.Eviction{
					LastUpdated: time.Now(),
				},
			}
		}

		states = append(states, &servicepodalerts.State{
			PodStates: podStates,
			ServiceState: &servicepodalerts.ServiceState{
				ServiceStatus:  "ACTIVE",
				SnapshotStatus: serviceStates[i],
			},
			Timestamp: time.Now(),
		})
	}

	return states
}

func generateInfo(podNames []string) *servicepodalerts.Info {
	podsInfo := make(map[string]*pods.PodInfo)
	for _, podName := range podNames {
		hostName := fmt.Sprintf("%s.sas.yp-c.yandex.net", podName)
		podsInfo[hostName] = &pods.PodInfo{
			PodID:    podName,
			Cluster:  "sas",
			NodeID:   "some-node-id",
			Itype:    "default",
			HostName: hostName,
		}
	}

	return &servicepodalerts.Info{
		PodsInfo: podsInfo,
	}
}

func main() {
	cfg := &servicepodalerts.GeneralTemplateConfig{
		NannyURL:                          "https://nanny.yandex-team.ru",
		DeployURL:                         "https://deploy.yandex-team.ru",
		PollPeriod:                        time.Hour * 4,
		MaxFaultyPodsTableSize:            10,
		MaxEvictionRequestedPodsTableSize: 10,
		ManualEvictionPeriod:              time.Hour * 24 * 4,
	}

	podsCount := 15

	podNames := []string{}
	for i := 1; i <= podsCount; i++ {
		podNames = append(podNames, fmt.Sprintf("test-bromigo-%d", i))
	}

	faultyPodNames := []string{}
	for _, pn := range podNames {
		faultyPodNames = append(faultyPodNames, fmt.Sprintf("%s.sas.yp-c.yandex.net", pn))
	}

	evictionRequestedPodNames := faultyPodNames

	states := generateStates(
		faultyPodNames,
		[]string{"ACTIVE", "HOOK_SEMI_FAILED", "PREPARED", "REMOVED", "ERROR"},
		[]pb.SnapshotStatus_Status{
			pb.SnapshotStatus_PREPARED,
			pb.SnapshotStatus_ACTIVE,
			pb.SnapshotStatus_PREPARING,
			pb.SnapshotStatus_DEACTIVATE_PENDING,
			pb.SnapshotStatus_GENERATING,
		},
	)

	decision := servicepodalerts.Decision{
		Pods: servicepodalerts.PodsDecision{
			Faulty:                    true,
			States:                    states,
			FaultyPodNames:            faultyPodNames,
			EvictionRequestedPodNames: evictionRequestedPodNames,
		},
		Service: servicepodalerts.ServiceDecision{
			States: states,
		},
	}

	info := generateInfo(podNames)

	tmpl := servicepodalerts.GeneralTemplateController{
		ServiceID: "test",
		Decision:  decision,
		Info:      info,
		Config:    cfg,
	}

	err := tmpl.Init()
	if err != nil {
		log.Fatalf(err.Error())
	}
	data, err := tmpl.Render()
	if err != nil {
		log.Fatalf(err.Error())
	}

	fd, err := os.Create(OutFile)
	if err != nil {
		log.Fatalf(err.Error())
	}
	defer fd.Close()

	_, err = fd.Write([]byte(data))
	if err != nil {
		log.Fatalf(err.Error())
	}
}
