package servicepodalerts

import (
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/infra/temporal/activities/abc"
	"a.yandex-team.ru/infra/temporal/activities/nanny/services"
	"a.yandex-team.ru/infra/temporal/clients/startrek"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
	"go.temporal.io/api/enums/v1"
	"go.temporal.io/sdk/workflow"
)

const (
	DutyInvocationText        = "Привет!\nНужна твоя помощь, как дежурного nanny"
	CloseReasonNowTestingText = "Вы пометили сервис как тестинг, закрываем тикет."
	CloseReasonAllOk          = "Для вашего сервиса больше нет уведомлений, закрываем тикет."
	CloseReasonServiceDeleted = "Ваш сервис удален, закрываем тикет."
)

type InvocationReason int

const (
	DutySchedule InvocationReason = iota
	ServiceRuntimeAttrsAuthor
	AbcMember
	RandomDutyFromService
	None
)

type TicketControllerConfig struct {
	NamespaceID                       string
	TaskQueue                         string
	Queue                             string
	Tags                              []string
	RetryInvocationPeriod             time.Duration
	NannyScheduleID                   int
	MaxResponsibleToSummon            int
	MaxFaultyPodsTableSize            int
	MaxEvictionRequestedPodsTableSize int
	ManualEvictionPeriod              time.Duration
	BudgetLeftThreshold               float64
	MinBudgetLeft                     int
	NannyURL                          string
	DeployURL                         string
	PollPeriod                        time.Duration
	EvictionExpiredThreshold          time.Duration
}

type InvocationData struct {
	Responsible *processor.Responsible
	InvocationReason
}

type StartrekerExecution struct {
	Workflow           *workflow.Execution
	LastInvocationData *InvocationData
}

type TicketController struct {
	serviceID           string
	ctx                 workflow.Context
	cfg                 *TicketControllerConfig
	startrekerExec      *StartrekerExecution
	serviceActivities   *services.Activities
	abcActivities       *abc.Activities
	timeoutConfig       *TimeoutConfig
	activitiesTaskQueue string
}

func (ctrl *TicketController) getServiceLatestRuntimeAttrsAuthors() ([]string, error) {
	ao := workflow.ActivityOptions{
		TaskQueue:           ctrl.activitiesTaskQueue,
		StartToCloseTimeout: ctrl.timeoutConfig.NannyRequestTimeout,
	}
	activityCtx := workflow.WithActivityOptions(ctrl.ctx, ao)
	var authors []string
	err := workflow.ExecuteActivity(
		activityCtx,
		ctrl.serviceActivities.GetServiceLatestRuntimeAttrsAuthors,
		ctrl.serviceID).Get(activityCtx, &authors)
	if err != nil {
		return nil, err
	}
	return authors, nil
}

func (ctrl *TicketController) getOnDutyFromService(serviceInfo *ServiceInfo) ([]string, error) {
	ao := workflow.ActivityOptions{
		TaskQueue:           ctrl.activitiesTaskQueue,
		StartToCloseTimeout: ctrl.timeoutConfig.AbcRequestTimeout,
	}
	activityCtx := workflow.WithActivityOptions(ctrl.ctx, ao)
	var logins []string
	err := workflow.ExecuteActivity(
		activityCtx,
		ctrl.abcActivities.GetOnDutyFromService,
		serviceInfo.InfoAttrs.AbcGroup).Get(activityCtx, &logins)
	if err != nil {
		return nil, err
	}
	return logins, nil
}

func (ctrl *TicketController) getAbcMembers(serviceInfo *ServiceInfo) ([]string, error) {
	ao := workflow.ActivityOptions{
		TaskQueue:           ctrl.activitiesTaskQueue,
		StartToCloseTimeout: ctrl.timeoutConfig.AbcRequestTimeout,
	}
	activityCtx := workflow.WithActivityOptions(ctrl.ctx, ao)

	var abcID int
	if serviceInfo.InfoAttrs.DutySchedule.ABCServiceID != 0 {
		abcID = serviceInfo.InfoAttrs.DutySchedule.ABCServiceID
	} else {
		abcID = serviceInfo.InfoAttrs.AbcGroup
	}

	var members []string
	err := workflow.ExecuteActivity(
		activityCtx,
		ctrl.abcActivities.GetRandomServiceMembers,
		abcID,
		ctrl.cfg.MaxResponsibleToSummon).Get(activityCtx, &members)
	if err != nil {
		return nil, err
	}
	return members, nil
}

func (ctrl *TicketController) commentTicket(message string) error {
	msg := processor.CommentProblemSignal{
		Comment: startrek.Comment{
			Text: message,
		},
	}
	childCtx := workflow.WithWorkflowNamespace(ctrl.ctx, ctrl.cfg.NamespaceID)
	err := workflow.SignalExternalWorkflow(
		childCtx,
		ctrl.startrekerExec.Workflow.ID,
		ctrl.startrekerExec.Workflow.RunID,
		"CommentProblem",
		msg).Get(ctrl.ctx, nil)
	return err
}

func (ctrl *TicketController) closeStartrekerWorkflowIfExists(reason string) error {
	if ctrl.startrekerExec != nil {
		err := ctrl.commentTicket(reason)
		if err != nil {
			return err
		}
		childCtx := workflow.WithWorkflowNamespace(ctrl.ctx, ctrl.cfg.NamespaceID)
		err = workflow.SignalExternalWorkflow(
			childCtx,
			ctrl.startrekerExec.Workflow.ID,
			ctrl.startrekerExec.Workflow.RunID,
			"CloseProblem",
			nil).Get(ctrl.ctx, nil)
		if err != nil {
			return err
		}
		ctrl.startrekerExec = nil
	}
	return nil
}

func (ctrl *TicketController) getServiceResponsible(serviceInfo *ServiceInfo) (*InvocationData, error) {
	dutySchedule := serviceInfo.InfoAttrs.DutySchedule
	if dutySchedule.ID != 0 {
		return &InvocationData{
			Responsible: &processor.Responsible{
				Kind:          processor.AbcSchedule,
				AbcScheduleID: dutySchedule.ID,
			},
			InvocationReason: DutySchedule,
		}, nil
	}

	onDuty, err := ctrl.getOnDutyFromService(serviceInfo)
	if err != nil {
		return nil, err
	}
	if len(onDuty) != 0 {
		return &InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: onDuty[:1],
			},
			InvocationReason: RandomDutyFromService,
		}, nil
	}

	authors, err := ctrl.getServiceLatestRuntimeAttrsAuthors()
	if err != nil {
		return nil, err
	}
	if len(authors) != 0 {
		return &InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: authors,
			},
			InvocationReason: ServiceRuntimeAttrsAuthor,
		}, nil
	}

	// always summon same abc members
	if ctrl.startrekerExec != nil {
		lastInvocationData := ctrl.startrekerExec.LastInvocationData
		if lastInvocationData.InvocationReason == AbcMember {
			return lastInvocationData, nil
		}
	}

	members, err := ctrl.getAbcMembers(serviceInfo)
	if err != nil {
		return nil, err
	}
	if len(members) != 0 {
		return &InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: members,
			},
			InvocationReason: AbcMember,
		}, nil
	}

	return nil, errors.New("no responsibe found")
}

func (ctrl *TicketController) getRetryInvocationSettings(decision *Decision, info *Info) *processor.RetryInvocationSettings {
	faultyPods := len(decision.Pods.FaultyPodNames)
	budget := info.ServiceInfo.MaxUnavailablePods
	budgetLeft := budget - faultyPods
	if budgetLeft < 0 {
		budgetLeft = 0
	}

	retryInvocationSettings := processor.RetryInvocationSettings{
		Kind:   processor.AckPeriod,
		Period: ctrl.cfg.RetryInvocationPeriod,
	}
	if info.ServiceInfo.MaxUnavailablePods == 0 {
		return &retryInvocationSettings
	}

	if budgetLeft <= ctrl.cfg.MinBudgetLeft && float64(budgetLeft)/float64(budget) < ctrl.cfg.BudgetLeftThreshold {
		return &retryInvocationSettings
	}

	if decision.Service.Faulty {
		return &retryInvocationSettings
	}

	if len(decision.Pods.EvictionRequestedPodNames) != 0 {
		return &retryInvocationSettings
	}

	return &processor.RetryInvocationSettings{
		Kind:   processor.Once,
		Period: ctrl.cfg.RetryInvocationPeriod,
	}
}

func (ctrl *TicketController) getProblemDescription(tmpl *GeneralTemplateController) (string, error) {
	return tmpl.Render()
}

func (ctrl *TicketController) getInvocationData(retryInvocationSettings *processor.RetryInvocationSettings, info *Info) (*InvocationData, error) {
	if retryInvocationSettings.Kind == processor.Once {
		return &InvocationData{
			Responsible: &processor.Responsible{
				Kind:   processor.Logins,
				Logins: []string{},
			},
			InvocationReason: None,
		}, nil
	}

	return ctrl.getServiceResponsible(info.ServiceInfo)
}

func (ctrl *TicketController) getInvocationText(invocationData *InvocationData, tmpl *GeneralTemplateController) (string, error) {
	var text string
	switch invocationData.InvocationReason {
	case DutySchedule:
		text = "Вы были выбраны, так как являетесь текущим дежурным в ABC сервисе, ((https://clubs.at.yandex-team.ru/infra-cloud/1955 который указан в няне)). "
	case RandomDutyFromService:
		text = "Вы были выбраны, так как являетесь дежурным этого сервиса. Из-за того, что у вас не ((https://clubs.at.yandex-team.ru/infra-cloud/1955 указан календарь дежурств)), мы выбрали случайный календарь из указанного в няне ABC-сервиса. "
	case ServiceRuntimeAttrsAuthor:
		text = "Вы были выбраны, так как недавно выкатывали изменения в этом сервисе. Это очень неточная эвристика, поэтому рекомендуем ((https://clubs.at.yandex-team.ru/infra-cloud/1955 указать календарь дежурств)). "
	case AbcMember:
		text = "Мы выбрали несколько человек из ABC сервиса, потому что не знаем, кто сможет этой проблемой заняться. Пожалуйста, ((https://clubs.at.yandex-team.ru/infra-cloud/1955 укажите календарь дежурств)). "
	}

	templateInvocationText := tmpl.GetInvocationText()

	return fmt.Sprintf("Обратите внимание на тикет. %s\n\n%s", text, templateInvocationText), nil
}

func (ctrl *TicketController) FormProblem(decision *Decision, info *Info, retryInvocationSettings *processor.RetryInvocationSettings, invocationData *InvocationData) (*processor.Problem, error) {
	generalTemplateController := &GeneralTemplateController{
		Config: &GeneralTemplateConfig{
			NannyScheduleID:                   ctrl.cfg.NannyScheduleID,
			NannyURL:                          ctrl.cfg.NannyURL,
			DeployURL:                         ctrl.cfg.DeployURL,
			PollPeriod:                        ctrl.cfg.PollPeriod,
			MaxFaultyPodsTableSize:            ctrl.cfg.MaxFaultyPodsTableSize,
			MaxEvictionRequestedPodsTableSize: ctrl.cfg.MaxEvictionRequestedPodsTableSize,
			ManualEvictionPeriod:              ctrl.cfg.ManualEvictionPeriod,
			EvictionExpiredThreshold:          ctrl.cfg.EvictionExpiredThreshold,
		},
		ServiceID:               ctrl.serviceID,
		Decision:                *decision,
		Info:                    info,
		RetryInvocationSettings: retryInvocationSettings,
	}
	err := generalTemplateController.Init()
	if err != nil {
		return nil, err
	}

	description, err := ctrl.getProblemDescription(generalTemplateController)
	if err != nil {
		return nil, err
	}

	invocationText, err := ctrl.getInvocationText(invocationData, generalTemplateController)
	if err != nil {
		return nil, err
	}

	problem := &processor.Problem{
		Ticket: startrek.Ticket{
			Summary:     fmt.Sprintf("Нотификации о сервисе %s", ctrl.serviceID),
			Description: description,
			Queue:       &startrek.Queue{Key: ctrl.cfg.Queue},
			Tags:        ctrl.cfg.Tags,
		},
		InvocationSettings: processor.InvocationSettings{
			Responsible:             *invocationData.Responsible,
			RetryInvocationSettings: *retryInvocationSettings,
			Text:                    invocationText,
		},
		InfraDutyInvocationSettings: &processor.InfraDutyInvocationSettings{
			AbcScheduleID: ctrl.cfg.NannyScheduleID,
			Text:          "",
		},
	}
	return problem, nil
}

func (ctrl *TicketController) CreateOrUpdateTicket(decision *Decision, info *Info) error {
	retryInvocationSettings := ctrl.getRetryInvocationSettings(decision, info)
	invocationData, err := ctrl.getInvocationData(retryInvocationSettings, info)
	if err != nil {
		return err
	}

	problem, err := ctrl.FormProblem(decision, info, retryInvocationSettings, invocationData)
	if err != nil {
		return err
	}

	if ctrl.startrekerExec == nil {
		cwo := workflow.ChildWorkflowOptions{
			Namespace:         ctrl.cfg.NamespaceID,
			WorkflowID:        fmt.Sprintf("pod-alerts-%s", ctrl.serviceID),
			ParentClosePolicy: enums.PARENT_CLOSE_POLICY_ABANDON,
			TaskQueue:         ctrl.cfg.TaskQueue,
		}
		ctx := workflow.WithChildOptions(ctrl.ctx, cwo)
		we := workflow.Execution{}
		err := workflow.ExecuteChildWorkflow(ctx, processor.ProcessWorkflow, *problem).GetChildWorkflowExecution().Get(ctx, &we)
		if err != nil {
			return err
		}

		ctrl.startrekerExec = &StartrekerExecution{Workflow: &we}
	} else {
		childCtx := workflow.WithWorkflowNamespace(ctrl.ctx, ctrl.cfg.NamespaceID)
		err := workflow.SignalExternalWorkflow(
			childCtx,
			ctrl.startrekerExec.Workflow.ID,
			ctrl.startrekerExec.Workflow.RunID,
			"UpdateProblem",
			processor.UpdateProblemSignal{Problem: *problem, Notify: false},
		).Get(ctrl.ctx, nil)
		if err != nil {
			return err
		}
	}

	if retryInvocationSettings.Kind == processor.AckPeriod {
		childCtx := workflow.WithWorkflowNamespace(ctrl.ctx, ctrl.cfg.NamespaceID)
		err := workflow.SignalExternalWorkflow(
			childCtx,
			ctrl.startrekerExec.Workflow.ID,
			ctrl.startrekerExec.Workflow.RunID,
			"AckSummon",
			nil,
		).Get(ctrl.ctx, nil)
		if err != nil {
			return err
		}
	}

	ctrl.startrekerExec.LastInvocationData = invocationData
	return nil
}
