package servicepodalerts

import (
	"context"
	"testing"
	"time"

	"a.yandex-team.ru/infra/temporal/activities/abc"
	"a.yandex-team.ru/infra/temporal/activities/io"
	"a.yandex-team.ru/infra/temporal/activities/nanny/services"
	client "a.yandex-team.ru/infra/temporal/clients/nanny"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	"go.temporal.io/sdk/testsuite"
	"go.temporal.io/sdk/workflow"
)

var serviceActivities *services.Activities
var abcActivities *abc.Activities
var ioActivities *io.Activities

type TicketTestSuite struct {
	suite.Suite
	testsuite.WorkflowTestSuite

	env *testsuite.TestWorkflowEnvironment
}

func (suite *TicketTestSuite) SetupTest() {
	suite.env = suite.NewTestWorkflowEnvironment()
}

func (suite *TicketTestSuite) AfterTest() {
	suite.env.AssertExpectations(suite.T())
}

func makeControllerWithoutContext() *TicketController {
	return &TicketController{
		serviceID: "test",
		cfg: &TicketControllerConfig{
			TaskQueue:              "TEST",
			MaxResponsibleToSummon: 5,
		},
		timeoutConfig: &TimeoutConfig{
			NannyRequestTimeout: time.Hour,
			AbcRequestTimeout:   time.Hour,
			YpRequestTimeout:    time.Hour,
		},
	}
}

func makeController(ctx workflow.Context) *TicketController {
	ctrl := makeControllerWithoutContext()
	ctrl.ctx = ctx
	return ctrl
}

// To get access to workflow context
func GetRuntimeAttrsWF(ctx workflow.Context) ([]string, error) {
	return makeController(ctx).getServiceLatestRuntimeAttrsAuthors()
}

func (suite *TicketTestSuite) TestGetRuntimeAttrsAuthors() {
	suite.env.OnActivity(
		serviceActivities.GetServiceLatestRuntimeAttrsAuthors,
		mock.Anything, mock.Anything,
	).Return(
		func(ctx context.Context, serviceID string) ([]string, error) {
			suite.Equal("test", serviceID)
			return []string{"author1", "author2"}, nil
		},
	)

	suite.env.ExecuteWorkflow(GetRuntimeAttrsWF)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var authors []string
	err = suite.env.GetWorkflowResult(&authors)
	suite.NoError(err)

	suite.EqualValues([]string{"author1", "author2"}, authors)
}

func GetOnDutyFromServiceWF(ctx workflow.Context, serviceInfo *ServiceInfo) ([]string, error) {
	return makeController(ctx).getOnDutyFromService(serviceInfo)
}

func (suite *TicketTestSuite) TestGetOnDutyFromService() {
	suite.env.OnActivity(
		abcActivities.GetOnDutyFromService,
		mock.Anything, mock.Anything,
	).Return(
		func(ctx context.Context, abcGroup int) ([]string, error) {
			suite.Equal(111, abcGroup)
			return []string{"login1", "login2"}, nil
		},
	)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			AbcGroup: 111,
		},
	}
	suite.env.ExecuteWorkflow(GetOnDutyFromServiceWF, serviceInfo)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var logins []string
	err = suite.env.GetWorkflowResult(&logins)
	suite.NoError(err)

	suite.EqualValues([]string{"login1", "login2"}, logins)
}

func GetAbcMembersWF(ctx workflow.Context, serviceInfo *ServiceInfo) ([]string, error) {
	return makeController(ctx).getAbcMembers(serviceInfo)
}

func (suite *TicketTestSuite) TestGetAbcMembersDutySchedule() {
	suite.env.OnActivity(
		abcActivities.GetRandomServiceMembers,
		mock.Anything, mock.Anything, mock.Anything,
	).Return(
		func(ctx context.Context, abcGroup int, maxResponsibleToSummon int) ([]string, error) {
			suite.Equal(222, abcGroup)
			suite.Equal(5, maxResponsibleToSummon)
			return []string{"login1", "login2"}, nil
		},
	)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			DutySchedule: client.ServiceDutySchedule{
				ABCServiceID: 222,
			},
			AbcGroup: 111,
		},
	}
	suite.env.ExecuteWorkflow(GetAbcMembersWF, serviceInfo)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var logins []string
	err = suite.env.GetWorkflowResult(&logins)
	suite.NoError(err)

	suite.EqualValues([]string{"login1", "login2"}, logins)
}

func (suite *TicketTestSuite) TestGetAbcMembersAbcGroup() {
	suite.env.OnActivity(
		abcActivities.GetRandomServiceMembers,
		mock.Anything, mock.Anything, mock.Anything,
	).Return(
		func(ctx context.Context, abcGroup int, maxResponsibleToSummon int) ([]string, error) {
			suite.Equal(111, abcGroup)
			return []string{"login1", "login2"}, nil
		},
	)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			AbcGroup: 111,
		},
	}
	suite.env.ExecuteWorkflow(GetAbcMembersWF, serviceInfo)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var logins []string
	err = suite.env.GetWorkflowResult(&logins)
	suite.NoError(err)

	suite.EqualValues([]string{"login1", "login2"}, logins)
}

func (suite *TicketTestSuite) TestGetServiceResponsibleDutySchedule() {
	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			DutySchedule: client.ServiceDutySchedule{
				ID: 123,
			},
		},
	}

	invocationData, err := makeControllerWithoutContext().getServiceResponsible(serviceInfo)
	suite.NoError(err)

	suite.Equal(invocationData.InvocationReason, DutySchedule)
	suite.Equal(invocationData.Responsible, &processor.Responsible{
		Kind:          processor.AbcSchedule,
		AbcScheduleID: 123,
	})
}

func GetServiceResponsibleWF(ctx workflow.Context, serviceInfo *ServiceInfo, prevData *InvocationData) (*InvocationData, error) {
	ctrl := makeController(ctx)
	if prevData != nil {
		ctrl.startrekerExec = &StartrekerExecution{
			LastInvocationData: prevData,
		}
	}
	return ctrl.getServiceResponsible(serviceInfo)
}

func (suite *TicketTestSuite) TestGetServiceResponsibleRandomDuty() {
	suite.env.OnActivity(abcActivities.GetOnDutyFromService, mock.Anything, mock.Anything).Return([]string{"login1", "login2"}, nil)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			DutySchedule: client.ServiceDutySchedule{
				ABCServiceID: 123,
			},
		},
	}
	suite.env.ExecuteWorkflow(GetServiceResponsibleWF, serviceInfo, nil)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var invocationData *InvocationData
	err = suite.env.GetWorkflowResult(&invocationData)
	suite.NoError(err)

	suite.Equal(invocationData.InvocationReason, RandomDutyFromService)
	suite.Equal(invocationData.Responsible, &processor.Responsible{
		Kind:   processor.Logins,
		Logins: []string{"login1"},
	})
}

func (suite *TicketTestSuite) TestGetServiceResponsibleRuntimeAttrsAuthors() {
	suite.env.OnActivity(abcActivities.GetOnDutyFromService, mock.Anything, mock.Anything).Return(nil, nil)
	suite.env.OnActivity(serviceActivities.GetServiceLatestRuntimeAttrsAuthors, mock.Anything, mock.Anything).Return([]string{"login"}, nil)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			DutySchedule: client.ServiceDutySchedule{
				ABCServiceID: 123,
			},
		},
	}
	suite.env.ExecuteWorkflow(GetServiceResponsibleWF, serviceInfo, nil)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var invocationData *InvocationData
	err = suite.env.GetWorkflowResult(&invocationData)
	suite.NoError(err)

	suite.Equal(invocationData.InvocationReason, ServiceRuntimeAttrsAuthor)
	suite.Equal(invocationData.Responsible, &processor.Responsible{
		Kind:   processor.Logins,
		Logins: []string{"login"},
	})
}

func (suite *TicketTestSuite) TestGetServiceResponsibleRandomPrevious() {
	suite.env.OnActivity(abcActivities.GetOnDutyFromService, mock.Anything, mock.Anything).Return(nil, nil)
	suite.env.OnActivity(serviceActivities.GetServiceLatestRuntimeAttrsAuthors, mock.Anything, mock.Anything).Return(nil, nil)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			DutySchedule: client.ServiceDutySchedule{
				ABCServiceID: 123,
			},
		},
	}

	lastInvocationData := &InvocationData{
		InvocationReason: AbcMember,
		Responsible: &processor.Responsible{
			Kind:   processor.Logins,
			Logins: []string{"login"},
		},
	}

	suite.env.ExecuteWorkflow(GetServiceResponsibleWF, serviceInfo, lastInvocationData)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var invocationData *InvocationData
	err = suite.env.GetWorkflowResult(&invocationData)
	suite.NoError(err)

	suite.Equal(invocationData, lastInvocationData)
}

func (suite *TicketTestSuite) TestGetServiceResponsibleRandomNew() {
	suite.env.OnActivity(abcActivities.GetOnDutyFromService, mock.Anything, mock.Anything).Return(nil, nil)
	suite.env.OnActivity(serviceActivities.GetServiceLatestRuntimeAttrsAuthors, mock.Anything, mock.Anything).Return(nil, nil)
	suite.env.OnActivity(abcActivities.GetRandomServiceMembers, mock.Anything, mock.Anything, mock.Anything).Return([]string{"login"}, nil)

	serviceInfo := &ServiceInfo{
		InfoAttrs: &client.ServiceInfoAttrs{
			DutySchedule: client.ServiceDutySchedule{
				ABCServiceID: 123,
			},
		},
	}

	suite.env.ExecuteWorkflow(GetServiceResponsibleWF, serviceInfo, nil)
	err := suite.env.GetWorkflowError()
	suite.NoError(err)

	var invocationData *InvocationData
	err = suite.env.GetWorkflowResult(&invocationData)
	suite.NoError(err)

	suite.Equal(invocationData, &InvocationData{
		InvocationReason: AbcMember,
		Responsible: &processor.Responsible{
			Kind:   processor.Logins,
			Logins: []string{"login"},
		},
	})
}

func GetRetryInvocationSettingsWF(ctx workflow.Context, decision *Decision, serviceInfo *ServiceInfo) (*processor.RetryInvocationSettings, error) {
	ctrl := makeController(ctx)
	ctrl.cfg.BudgetLeftThreshold = 0.8
	ctrl.cfg.MinBudgetLeft = 10
	return ctrl.getRetryInvocationSettings(decision, &Info{ServiceInfo: serviceInfo}), nil
}

func (suite *TicketTestSuite) TestRetryInvocationSettingsThresholdOk() {
	// minBudgetLeft exceeded, but threshold isn't reached
	faultyPodsCount := 1
	decision := &Decision{
		Pods: PodsDecision{},
	}
	for i := 0; i < faultyPodsCount; i += 1 {
		decision.Pods.FaultyPodNames = append(decision.Pods.FaultyPodNames, "test")
	}

	serviceInfo := &ServiceInfo{
		MaxUnavailablePods: 10,
	}

	suite.env.ExecuteWorkflow(GetRetryInvocationSettingsWF, decision, serviceInfo)
	invocationSettings := processor.RetryInvocationSettings{}
	err := suite.env.GetWorkflowResult(&invocationSettings)
	suite.NoError(err)

	suite.Equal(processor.Once, invocationSettings.Kind)
}

func (suite *TicketTestSuite) TestRetryInvocationSettingsBudgetOk() {
	// threshold is reached, but minBudgetLeft is not
	faultyPodsCount := 30
	decision := &Decision{
		Pods: PodsDecision{},
	}
	for i := 0; i < faultyPodsCount; i += 1 {
		decision.Pods.FaultyPodNames = append(decision.Pods.FaultyPodNames, "test")
	}

	serviceInfo := &ServiceInfo{
		MaxUnavailablePods: 100,
	}

	suite.env.ExecuteWorkflow(GetRetryInvocationSettingsWF, decision, serviceInfo)
	invocationSettings := processor.RetryInvocationSettings{}
	err := suite.env.GetWorkflowResult(&invocationSettings)
	suite.NoError(err)

	suite.Equal(processor.Once, invocationSettings.Kind)
}

func (suite *TicketTestSuite) TestRetryInvocationSettingsSharded() {
	faultyPodsCount := 1
	decision := &Decision{
		Pods: PodsDecision{},
	}
	for i := 0; i < faultyPodsCount; i += 1 {
		decision.Pods.FaultyPodNames = append(decision.Pods.FaultyPodNames, "test")
	}

	serviceInfo := &ServiceInfo{
		// for sharded service max unavailable will be 0
		MaxUnavailablePods: 0,
	}

	suite.env.ExecuteWorkflow(GetRetryInvocationSettingsWF, decision, serviceInfo)
	invocationSettings := processor.RetryInvocationSettings{}
	err := suite.env.GetWorkflowResult(&invocationSettings)
	suite.NoError(err)

	suite.Equal(processor.AckPeriod, invocationSettings.Kind)
}

func (suite *TicketTestSuite) TestRetryInvocationSettingsFaulty() {
	faultyPodsCount := 3
	decision := &Decision{
		Pods: PodsDecision{},
	}
	for i := 0; i < faultyPodsCount; i += 1 {
		decision.Pods.FaultyPodNames = append(decision.Pods.FaultyPodNames, "test")
	}

	serviceInfo := &ServiceInfo{
		MaxUnavailablePods: 10,
	}

	suite.env.ExecuteWorkflow(GetRetryInvocationSettingsWF, decision, serviceInfo)
	invocationSettings := processor.RetryInvocationSettings{}
	err := suite.env.GetWorkflowResult(&invocationSettings)
	suite.NoError(err)

	suite.Equal(processor.AckPeriod, invocationSettings.Kind)
}

func CreateOrUpdateTicketWF(ctx workflow.Context, decision *Decision, info *Info) (*TicketController, error) {
	ctrl := makeController(ctx)
	err := ctrl.CreateOrUpdateTicket(decision, info)
	if err != nil {
		return nil, err
	}
	return ctrl, nil
}

func (suite *TicketTestSuite) TestCreateOrUpdateTicket() {

}

func TestTicketTestSuite(t *testing.T) {
	suite.Run(t, new(TicketTestSuite))
}
