package processor

import (
	"fmt"
	"reflect"
	"time"

	"a.yandex-team.ru/infra/temporal/activities/startreker"
	"a.yandex-team.ru/infra/temporal/clients/startrek"

	"go.temporal.io/sdk/workflow"
)

func processWorkflowV1(ctx workflow.Context, problem Problem) error {
	l := workflow.GetLogger(ctx)
	var a *startreker.Activities
	var key startrek.TicketKey
	wfID := workflow.GetInfo(ctx).WorkflowExecution.ID
	problem.Ticket.MacroTag = wfID
	err := workflow.ExecuteActivity(ctx, a.CreateTicket, problem.Ticket).Get(ctx, &key)
	if err != nil {
		return fmt.Errorf("failed to create ticket: %w", err)
	}
	l.Info("created ticket:", key)

	closed := false
	s := workflow.NewSelector(ctx)
	updateSignalChan := workflow.GetSignalChannel(ctx, "UpdateProblem")
	s.AddReceive(updateSignalChan, func(c workflow.ReceiveChannel, more bool) {
		var signalVal UpdateProblemSignal
		c.Receive(ctx, &signalVal)
		if !reflect.DeepEqual(signalVal.Problem.Ticket, problem.Ticket) {
			err := workflow.ExecuteActivity(ctx, a.UpdateTicket, key, &signalVal.Problem.Ticket, signalVal.Notify).Get(ctx, nil)
			if err != nil {
				l.Error("failed to update ticket:", err)
			} else {
				l.Info("updated ticket:", key)
			}
		}
		problem = signalVal.Problem
		problem.Ticket.MacroTag = wfID
	})

	commentSignalChan := workflow.GetSignalChannel(ctx, "CommentProblem")
	s.AddReceive(commentSignalChan, func(c workflow.ReceiveChannel, more bool) {
		var signalVal CommentProblemSignal
		c.Receive(ctx, &signalVal)
		err := workflow.ExecuteActivity(ctx, a.CommentTicket, key, signalVal.Comment).Get(ctx, nil)
		if err != nil {
			l.Error("failed to comment ticket:", key)
		} else {
			l.Info("commented ticket:", key)
		}
	})

	closeSignalChan := workflow.GetSignalChannel(ctx, "CloseProblem")
	s.AddReceive(closeSignalChan, func(c workflow.ReceiveChannel, more bool) {
		c.Receive(ctx, nil)
		err := workflow.ExecuteActivity(ctx, a.CloseTicket, key).Get(ctx, nil)
		if err != nil {
			l.Error("failed to close ticket:", err)
		} else {
			l.Info("closed ticket:", key)
		}
		closed = true
	})

	summonInfraDutySignalChan := workflow.GetSignalChannel(ctx, "SummonInfraDuty")
	s.AddReceive(summonInfraDutySignalChan, func(c workflow.ReceiveChannel, more bool) {
		c.Receive(ctx, nil)
		if problem.InfraDutyInvocationSettings == nil {
			l.Warn("trying to summon infra duty to problem without InfraDutyInvocationSettings")
			return
		}

		comment := startrek.Comment{Text: problem.InfraDutyInvocationSettings.Text}
		err := workflow.ExecuteActivity(ctx, a.GetOnDuty, problem.InfraDutyInvocationSettings.AbcScheduleID).Get(ctx, &comment.Summonees)
		if err != nil {
			l.Error("failed to get service duty:", err)
			return
		}
		err = workflow.ExecuteActivity(ctx, a.CommentTicket, key, comment).Get(ctx, nil)
		if err != nil {
			l.Error("failed to comment ticket:", err)
			return
		}
	})

	var futureErr error

	if problem.InvocationSettings.RetryInvocationSettings.Kind == Period {
		var addTimerFutureHandler func(timerFuture workflow.Future)
		addTimerFutureHandler = func(timerFuture workflow.Future) {
			s.AddFuture(timerFuture, func(f workflow.Future) {
				err := Summon(ctx, a, problem, key)
				if err != nil {
					futureErr = err
				} else {
					newTimerFuture := workflow.NewTimer(ctx, problem.InvocationSettings.RetryInvocationSettings.Period)
					addTimerFutureHandler(newTimerFuture)
				}
			})
		}

		timerFuture := workflow.NewTimer(ctx, time.Second)
		addTimerFutureHandler(timerFuture)
	}

	future := workflow.ExecuteActivity(ctx, a.WaitUntilTicketClosed, key)
	s.AddFuture(future, func(f workflow.Future) {
		var isTicketClosed bool
		err := f.Get(ctx, &isTicketClosed)
		if err != nil {
			futureErr = err
		} else if isTicketClosed {
			closed = true
		}
	})

	for !closed {
		s.Select(ctx)
		if futureErr != nil {
			return futureErr
		}
	}
	return nil
}
