package processor

import (
	"fmt"
	"reflect"
	"time"

	"a.yandex-team.ru/infra/temporal/activities/calendar"
	"a.yandex-team.ru/infra/temporal/activities/startreker"
	"a.yandex-team.ru/infra/temporal/clients/startrek"

	"go.temporal.io/sdk/log"
	"go.temporal.io/sdk/workflow"
	"k8s.io/apimachinery/pkg/util/wait"
)

func isOkToContinue(ctx workflow.Context) (bool, error) {
	var isWorkingDay bool
	err := workflow.ExecuteActivity(ctx, calendar.IsTodayWorkingDay).Get(ctx, &isWorkingDay)
	if err != nil {
		return false, fmt.Errorf("failed to check whether it's a working day: %w", err)
	}
	if !isWorkingDay {
		return false, fmt.Errorf("it's not a working day, not proceeding")
	}

	var isWorkingHours bool
	err = workflow.ExecuteActivity(ctx, calendar.IsNowWorkingHours).Get(ctx, &isWorkingHours)
	if err != nil {
		return false, fmt.Errorf("failed to check whether it's working hours: %w", err)
	}
	if !isWorkingHours {
		return false, fmt.Errorf("it's not working hours, not proceeding")
	}
	return true, nil
}

type Processor struct {
	ctx     workflow.Context
	problem Problem
	s       workflow.Selector
	key     startrek.TicketKey
	l       log.Logger
	a       *startreker.Activities
	wfID    string

	err                   error
	hasBeenSummoned       bool
	closed                bool
	isPeriodSummonRunning bool
	awaitsSummoning       bool
}

func newProcessor(ctx workflow.Context, problem Problem, l log.Logger, startrekKey startrek.TicketKey) (*Processor, error) {
	p := Processor{
		ctx:     ctx,
		l:       l,
		problem: problem,

		closed:                false,
		hasBeenSummoned:       false,
		isPeriodSummonRunning: false,
	}

	p.wfID = workflow.GetInfo(ctx).WorkflowExecution.ID
	p.problem.Ticket.MacroTag = p.wfID
	if startrekKey == "" {
		err := workflow.ExecuteActivity(ctx, p.a.CreateTicket, p.problem.Ticket).Get(ctx, &p.key)
		if err != nil {
			return nil, fmt.Errorf("failed to create ticket: %w", err)
		}
		p.l.Info("created ticket:", p.key)
	} else {
		p.key = startrekKey
	}

	return &p, nil
}

func (p *Processor) periodSummon(ctx workflow.Context) {
	if p.isPeriodSummonRunning {
		return
	}
	p.isPeriodSummonRunning = true
	for {
		kind := p.problem.InvocationSettings.RetryInvocationSettings.Kind
		if kind != Period && kind != AckPeriod {
			p.isPeriodSummonRunning = false
			return
		}

		if p.hasBeenSummoned && !p.problem.InvocationSettings.RetryInvocationSettings.SummonOnNonWorkingHours {
			if ok, err := isOkToContinue(ctx); !ok {
				p.l.Info("can not be summoned, wait: ", err)
				err = workflow.Sleep(ctx, time.Hour+wait.Jitter(time.Minute*2, 1.0))
				if err != nil {
					p.err = err
					return
				}
				continue
			}
		}

		if kind == AckPeriod {
			p.awaitsSummoning = true
		} else {
			err := Summon(ctx, p.a, p.problem, p.key)
			if err != nil {
				p.err = err
			} else {
				p.hasBeenSummoned = true
			}
		}

		err := workflow.Sleep(ctx, p.problem.InvocationSettings.RetryInvocationSettings.Period)
		if err != nil {
			p.err = err
			return
		}
	}
}

func (p *Processor) onUpdateProblem(problem Problem, notify bool) {
	if !reflect.DeepEqual(problem.Ticket, p.problem.Ticket) {
		err := workflow.ExecuteActivity(p.ctx, p.a.UpdateTicket, p.key, &problem.Ticket, notify).Get(p.ctx, nil)
		if err != nil {
			p.l.Error("failed to update ticket:", err)
		} else {
			p.l.Info("updated ticket:", p.key)
		}
	}

	newKind := problem.InvocationSettings.RetryInvocationSettings.Kind
	if p.problem.InvocationSettings.RetryInvocationSettings.Kind != newKind {
		if newKind == AckPeriod {
			p.awaitsSummoning = false
		}

		if newKind == Period || newKind == AckPeriod {
			workflow.Go(p.ctx, p.periodSummon)
		}
	}

	p.problem = problem
}

func (p *Processor) onCloseProblem() {
	err := workflow.ExecuteActivity(p.ctx, p.a.CloseTicket, p.key).Get(p.ctx, nil)
	if err != nil {
		p.l.Error("failed to close ticket:", err)
	} else {
		p.l.Info("closed ticket:", p.key)
	}
	p.closed = true
}

func (p *Processor) onCommentProblem(comment startrek.Comment) {
	err := workflow.ExecuteActivity(p.ctx, p.a.CommentTicket, p.key, comment).Get(p.ctx, nil)
	if err != nil {
		p.l.Error("failed to comment ticket:", p.key)
	} else {
		p.l.Info("commented ticket:", p.key)
	}
}

func (p *Processor) onSummonInfraDuty() {
	if p.problem.InfraDutyInvocationSettings == nil {
		p.l.Warn("trying to summon infra duty to problem without InfraDutyInvocationSettings")
		return
	}

	comment := startrek.Comment{Text: p.problem.InfraDutyInvocationSettings.Text}
	err := workflow.ExecuteActivity(p.ctx, p.a.GetOnDuty, p.problem.InfraDutyInvocationSettings.AbcScheduleID).Get(p.ctx, &comment.Summonees)
	if err != nil {
		p.l.Error("failed to get service duty:", err)
		return
	}
	err = workflow.ExecuteActivity(p.ctx, p.a.CommentTicket, p.key, comment).Get(p.ctx, nil)
	if err != nil {
		p.l.Error("failed to comment ticket:", err)
		return
	}
}

func (p *Processor) onAckSummon() {
	if !p.awaitsSummoning {
		return
	}

	err := Summon(p.ctx, p.a, p.problem, p.key)
	if err != nil {
		p.err = err
	} else {
		p.hasBeenSummoned = true
		p.awaitsSummoning = false
	}
}

func (p *Processor) close() {
	p.closed = true
}

func (p *Processor) panic(err error) {
	if err == nil {
		p.l.Warn("Trying to panic nil error")
		return
	}
	p.err = err
}

func (p *Processor) start() {
	switch p.problem.InvocationSettings.RetryInvocationSettings.Kind {
	case Period, AckPeriod:
		workflow.Go(p.ctx, p.periodSummon)
	case Once:
		err := Summon(p.ctx, p.a, p.problem, p.key)
		if err != nil {
			p.err = err
		}
	}
}

func (p *Processor) run() error {
	for !p.closed {
		p.s.Select(p.ctx)
		if p.err != nil {
			return p.err
		}
	}
	return nil
}

func (p *Processor) addCommentProblemReceiver() {
	commentSignalChan := workflow.GetSignalChannel(p.ctx, "CommentProblem")
	p.s.AddReceive(commentSignalChan, func(c workflow.ReceiveChannel, more bool) {
		var signalVal CommentProblemSignal
		c.Receive(p.ctx, &signalVal)
		p.onCommentProblem(signalVal.Comment)
	})
}

func (p *Processor) addCloseProblemReceiver() {
	closeSignalChan := workflow.GetSignalChannel(p.ctx, "CloseProblem")
	p.s.AddReceive(closeSignalChan, func(c workflow.ReceiveChannel, more bool) {
		c.Receive(p.ctx, nil)
		p.onCloseProblem()
	})
}

func (p *Processor) addSummonInfraDutyReceiver() {
	summonInfraDutySignalChan := workflow.GetSignalChannel(p.ctx, "SummonInfraDuty")
	p.s.AddReceive(summonInfraDutySignalChan, func(c workflow.ReceiveChannel, more bool) {
		c.Receive(p.ctx, nil)
		p.onSummonInfraDuty()
	})
}

func (p *Processor) addUpdateProblemReceiver() {
	future := workflow.ExecuteActivity(p.ctx, p.a.WaitUntilTicketClosed, p.key)
	p.s.AddFuture(future, func(f workflow.Future) {
		var isTicketClosed bool
		err := f.Get(p.ctx, &isTicketClosed)
		if err != nil {
			p.panic(err)
		} else if isTicketClosed {
			p.close()
		}
	})
}

func (p *Processor) addAckSummonReceiver() {
	ackSummonSignalChan := workflow.GetSignalChannel(p.ctx, "AckSummon")
	p.s.AddReceive(ackSummonSignalChan, func(c workflow.ReceiveChannel, more bool) {
		c.Receive(p.ctx, nil)
		p.onAckSummon()
	})
}

func (p *Processor) addWaitUntilTicketClosedProcessor() {
	updateSignalChan := workflow.GetSignalChannel(p.ctx, "UpdateProblem")
	p.s.AddReceive(updateSignalChan, func(c workflow.ReceiveChannel, more bool) {
		var signalVal UpdateProblemSignal
		c.Receive(p.ctx, &signalVal)
		p.onUpdateProblem(signalVal.Problem, signalVal.Notify)
	})
}

func processWorkflowV2(ctx workflow.Context, problem Problem, startrekKey startrek.TicketKey) error {
	l := workflow.GetLogger(ctx)
	p, err := newProcessor(ctx, problem, l, startrekKey)
	if err != nil {
		return err
	}

	p.s = workflow.NewSelector(ctx)
	p.addCommentProblemReceiver()
	p.addCloseProblemReceiver()
	p.addSummonInfraDutyReceiver()
	p.addUpdateProblemReceiver()
	p.addWaitUntilTicketClosedProcessor()
	p.addAckSummonReceiver()
	p.start()
	return p.run()
}
