package gc

import (
	acts "a.yandex-team.ru/infra/temporal/activities/yd/ri"
	swatyp "a.yandex-team.ru/infra/temporal/clients/yp"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"fmt"
	"go.temporal.io/api/enums/v1"
	"go.temporal.io/sdk/temporal"
	"go.temporal.io/sdk/workflow"
	"k8s.io/client-go/tools/cache"
	"strings"
	"time"
)

const (
	releasesFilter string = "[/status/processing/finished/status] = \"true\" AND [/status/progress/closed/status] = \"true\""
)

var releasesSelectors = []string{"/meta/id", "/status/processing", "/status/progress"}
var deployTicketsReleaseIDSelectors = []string{"/meta/id", "/spec/release_id"}

func makeDeployTicketsByReleaseIDsFilter(releaseIDs []string) string {
	if len(releaseIDs) == 0 {
		return ""
	}
	return fmt.Sprintf("[/spec/release_id] IN (\"%s\")", strings.Join(releaseIDs, "\", \""))
}

func newReleasesIndexer() cache.Indexer {
	return cache.NewIndexer(
		func(obj interface{}) (string, error) {
			return obj.(*ypapi.TRelease).GetMeta().GetId(), nil
		},
		cache.Indexers{})
}

func fillReleasesIndexer(activityCtx workflow.Context, a *acts.Activities, filter string, selectors []string, batchSize int, batchesCount int, continuationToken string, idxr cache.Indexer) (string, error) {
	ct := continuationToken
	for i := 0; i < batchesCount; i++ {
		var b swatyp.ReleasesBatch
		err := workflow.ExecuteActivity(
			activityCtx,
			a.SelectReleasesBatchActivity,
			filter,
			selectors,
			batchSize,
			ct,
		).Get(activityCtx, &b)
		if err != nil {
			return "", fmt.Errorf("executing SelectReleasesBatchActivity failed: %w", err)
		}
		for _, item := range b.Items {
			if err := idxr.Add(item); err != nil {
				return "", fmt.Errorf("failed to add release to indexer: %w", err)
			}
		}
		if len(b.Items) < batchSize {
			break
		}
		ct = b.ContinuationToken
	}
	return ct, nil
}

func fillDeployTicketsIndexerByReleaseIDs(activityCtx workflow.Context, a *acts.Activities, releaseIDs []string, batchSize int, tickets cache.Indexer) error {
	for _, b := range byBatches(releaseIDs, batchSize) {
		f := makeDeployTicketsByReleaseIDsFilter(b)
		continuationToken := ""
		for {
			ct, size, err := fillDeployTicketsBatchIndexer(activityCtx, a, f, deployTicketsReleaseIDSelectors, batchSize, continuationToken, tickets)
			if err != nil {
				return fmt.Errorf("fillDeployTicketsBatchIndexer failed: %w", err)
			}
			if size < batchSize {
				break
			}
			continuationToken = ct
		}
	}
	return nil
}

func isReleaseCandidateToRemove(r *ypapi.TRelease, ttl time.Duration) bool {
	s := r.GetStatus()
	c := s.GetProgress().GetClosed()
	if c.GetStatus() != ypapi.EConditionStatus_CS_TRUE {
		return false
	}
	if s.GetProcessing().GetFinished().GetStatus() != ypapi.EConditionStatus_CS_TRUE {
		return false
	}
	return c.GetLastTransitionTime().AsTime().Add(ttl).Before(time.Now())
}

func hasDeployTickets(r *ypapi.TRelease, tickets cache.Indexer) bool {
	tt, err := tickets.ByIndex("releaseID", r.GetMeta().GetId())
	if err != nil {
		return false
	}
	return len(tt) != 0
}

func ReleasesPortionGCWorkflow(ctx workflow.Context, config *Config, continuationToken string) error {
	logger := workflow.GetLogger(ctx)
	logger.Info(fmt.Sprintf("starting releases gc workflow for portion %s", continuationToken))

	options := workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 10,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.0,
			MaximumAttempts:        5,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	}

	activityCtx := workflow.WithActivityOptions(ctx, options)
	var a *acts.Activities

	logger.Info("fetching closed releases portion...")
	releases := newReleasesIndexer()
	nextContinuationToken, err := fillReleasesIndexer(activityCtx, a, releasesFilter, releasesSelectors, config.ReleasesBatchSize, config.ReleasesBatchesCountInPortion, continuationToken, releases)
	if err != nil {
		return fmt.Errorf("fillReleasesIndexer failed: %w", err)
	}
	logger.Info(fmt.Sprintf("closed releases count in portion: %d", len(releases.List())))

	candidates := make([]string, 0, len(releases.List()))
	for _, o := range releases.List() {
		r, ok := o.(*ypapi.TRelease)
		if !ok {
			logger.Warn("failed to cast stored object to TRelease")
			continue
		}
		if isReleaseCandidateToRemove(r, config.ReleaseTTL) {
			candidates = append(candidates, r.GetMeta().GetId())
		}
	}
	logger.Info(fmt.Sprintf("found candidates to remove: %d", len(candidates)))

	tickets := newDeployTicketsIndexer(cache.Indexers{"releaseID": indexDeployTicketsByReleaseID})
	logger.Info(fmt.Sprintf("fetching tickets for %d releases", len(candidates)))
	if err := fillDeployTicketsIndexerByReleaseIDs(activityCtx, a, candidates, config.DeployTicketsBatchSize, tickets); err != nil {
		return fmt.Errorf("fillDeployTicketsIndexerByReleaseIDs failed: %w", err)
	}
	toRemove := make([]string, 0, len(candidates))
	for _, rID := range candidates {
		o, exists, err := releases.GetByKey(rID)
		if !exists {
			logger.Warn(fmt.Sprintf("release not found in indexer: %s", rID))
			continue
		}
		if err != nil {
			logger.Warn(fmt.Sprintf("failed to get release from indexer: %s: %s", rID, err))
			continue
		}
		r, ok := o.(*ypapi.TRelease)
		if !ok {
			logger.Warn(fmt.Sprintf("failed to cast release from stored objects: %s", rID))
			continue
		}
		if !hasDeployTickets(r, tickets) {
			toRemove = append(toRemove, rID)
		}
	}
	logger.Info(fmt.Sprintf("releases to remove count: %d", len(toRemove)))
	logger.Info(fmt.Sprintf("releases to remove: %s", toRemove))
	for _, b := range byBatches(toRemove, config.ReleasesRemoveBatchSize) {
		err := workflow.ExecuteActivity(
			activityCtx,
			a.RemoveReleasesByIDsActivity,
			b,
		).Get(activityCtx, nil)
		if err != nil {
			return fmt.Errorf("failed to execute RemoveReleasesByIDsActivity: %w", err)
		}
	}
	logger.Info("workflow completed")
	portionSize := config.ReleasesBatchSize * config.ReleasesBatchesCountInPortion
	if len(releases.List()) < portionSize {
		logger.Info("releases gc workflow completed totally!")
		return nil
	}
	return workflow.NewContinueAsNewError(ctx, ReleasesPortionGCWorkflow, config, nextContinuationToken)
}

func CronReleasesGCWorkflow(ctx workflow.Context, config *Config) error {
	cwo := workflow.ChildWorkflowOptions{
		WorkflowID:        "releasesPortionGC",
		ParentClosePolicy: enums.PARENT_CLOSE_POLICY_ABANDON,
	}
	childCtx := workflow.WithChildOptions(ctx, cwo)
	err := workflow.ExecuteChildWorkflow(childCtx, ReleasesPortionGCWorkflow, config, "").GetChildWorkflowExecution().Get(childCtx, nil)
	return err
}
