package gc

import (
	acts "a.yandex-team.ru/infra/temporal/activities/yd/ri"
	swatyp "a.yandex-team.ru/infra/temporal/clients/yp"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"fmt"
	"go.temporal.io/api/enums/v1"
	"go.temporal.io/sdk/log"
	"go.temporal.io/sdk/temporal"
	"go.temporal.io/sdk/workflow"
	"k8s.io/client-go/tools/cache"
	"time"
)

const (
	deployTicketsFilter string = "[/status/progress/closed/status] = \"true\""
)

var deployTicketsSelectors = []string{"/meta/id", "/meta/stage_id", "/spec/release_id", "/status/progress"}

// to implement set
type void struct{}

var empty void

func indexDeployTicketsByStageID(obj interface{}) ([]string, error) {
	t, ok := obj.(*ypapi.TDeployTicket)
	if !ok {
		return []string{}, nil
	}
	return []string{t.GetMeta().GetStageId()}, nil
}

func indexDeployTicketsByReleaseID(obj interface{}) ([]string, error) {
	t, ok := obj.(*ypapi.TDeployTicket)
	if !ok {
		return []string{}, nil
	}
	return []string{t.GetSpec().GetReleaseId()}, nil
}

func newDeployTicketsIndexer(indexers cache.Indexers) cache.Indexer {
	return cache.NewIndexer(
		func(obj interface{}) (string, error) {
			return obj.(*ypapi.TDeployTicket).GetMeta().GetId(), nil
		},
		indexers)
}

func fillDeployTicketsBatchIndexer(activityCtx workflow.Context, a *acts.Activities, filter string, selectors []string, batchSize int, continuationToken string, idxr cache.Indexer) (string, int, error) {
	var b swatyp.DeployTicketsBatch
	err := workflow.ExecuteActivity(
		activityCtx,
		a.SelectDeployTicketsBatchActivity,
		filter,
		selectors,
		batchSize,
		continuationToken,
	).Get(activityCtx, &b)
	if err != nil {
		return "", 0, fmt.Errorf("executing SelectDeployTicketsBatchActivity failed: %w", err)
	}
	for _, item := range b.Items {
		if err := idxr.Add(item); err != nil {
			return "", 0, fmt.Errorf("failed to add ticket to indexer: %w", err)
		}
	}
	return b.ContinuationToken, len(b.Items), nil
}

func fillDeployTicketsIndexer(activityCtx workflow.Context, a *acts.Activities, filter string, selectors []string, batchSize int, batchesCount int, continuationToken string, idxr cache.Indexer) (string, error) {
	for i := 0; i < batchesCount; i++ {
		ct, size, err := fillDeployTicketsBatchIndexer(activityCtx, a, filter, selectors, batchSize, continuationToken, idxr)
		if err != nil {
			return "", fmt.Errorf("fillDeployTicketsBatchIndexer failed: %w", err)
		}
		if size < batchSize {
			return ct, nil
		}
		continuationToken = ct
	}
	return continuationToken, nil
}

func isDeployTicketReadyToRemove(t *ypapi.TDeployTicket, ttl time.Duration, soxTTL time.Duration, stages cache.Indexer, logger log.Logger) bool {
	c := t.GetStatus().GetProgress().GetClosed()
	if c.GetStatus() != ypapi.EConditionStatus_CS_TRUE {
		return false
	}
	if c.GetLastTransitionTime().AsTime().Add(ttl).After(time.Now()) {
		return false
	}
	o, exists, err := stages.GetByKey(t.GetMeta().GetStageId())
	if !exists {
		return false
	}
	if err != nil {
		return false
	}
	s, ok := o.(*ypapi.TStage)
	if !ok {
		return false
	}
	if s.GetSpec().GetSoxService() && c.GetLastTransitionTime().AsTime().Add(soxTTL).After(time.Now()) {
		return false
	}
	return true
}

func DeployTicketsPortionGCWorkflow(ctx workflow.Context, config *Config, continuationToken string) error {
	logger := workflow.GetLogger(ctx)
	logger.Info(fmt.Sprintf("starting tickets gc workflow for portion %s", continuationToken))

	options := workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 10,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.0,
			MaximumAttempts:        5,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	}

	activityCtx := workflow.WithActivityOptions(ctx, options)
	var a *acts.Activities

	logger.Info("fetching closed tickets portion...")
	ticketsIdxr := newDeployTicketsIndexer(cache.Indexers{"stageID": indexDeployTicketsByStageID, "releaseID": indexDeployTicketsByReleaseID})
	nextContinuationToken, err := fillDeployTicketsIndexer(activityCtx, a, deployTicketsFilter, deployTicketsSelectors, config.DeployTicketsBatchSize, config.DeployTicketsBatchesCountInPortion, continuationToken, ticketsIdxr)
	if err != nil {
		return fmt.Errorf("fillDeployTicketsIndexer failed: %w", err)
	}
	logger.Info(fmt.Sprintf("closed tickets count in portion: %d", len(ticketsIdxr.List())))

	sm := make(map[string]void, 1000)
	tickets := make([]*ypapi.TDeployTicket, 0, len(ticketsIdxr.List()))
	for _, o := range ticketsIdxr.List() {
		t, ok := o.(*ypapi.TDeployTicket)
		if !ok {
			logger.Warn("failed to cast stored object to TDeployTicket")
			continue
		}
		if t.GetStatus().GetProgress().GetClosed().GetStatus() != ypapi.EConditionStatus_CS_TRUE {
			continue
		}
		tickets = append(tickets, t)
		sm[t.GetMeta().GetStageId()] = empty
	}
	stageIDs := make([]string, 0, len(sm))
	for sID := range sm {
		stageIDs = append(stageIDs, sID)
	}
	stages := newStagesIndexer()
	logger.Info(fmt.Sprintf("fetching %d stages", len(stageIDs)))
	if err := fillStagesIndexer(activityCtx, a, stageIDs, config.StagesBatchSize, stages); err != nil {
		return fmt.Errorf("fillStagesIndexer failed: %w", err)
	}

	toRemove := make([]string, 0, len(tickets))
	for _, t := range tickets {
		if isDeployTicketReadyToRemove(t, config.DeployTicketTTL, config.SOXDeployTicketTTL, stages, logger) {
			toRemove = append(toRemove, t.GetMeta().GetId())
		}
	}
	logger.Info(fmt.Sprintf("tickets to remove count: %d", len(toRemove)))
	logger.Info(fmt.Sprintf("tickets to remove: %s", toRemove))
	for _, b := range byBatches(toRemove, config.DeployTicketsRemoveBatchSize) {
		err := workflow.ExecuteActivity(
			activityCtx,
			a.RemoveDeployTicketsByIDsActivity,
			b,
		).Get(activityCtx, nil)
		if err != nil {
			return fmt.Errorf("failed to execute RemoveDeployTicketsByIDsActivity: %w", err)
		}
	}
	logger.Info(fmt.Sprintf("tickets gc workflow completed for portion %s", continuationToken))
	portionSize := config.DeployTicketsBatchSize * config.DeployTicketsBatchesCountInPortion
	if len(ticketsIdxr.List()) < portionSize {
		logger.Info("tickets gc workflow completed totally!")
		return nil
	}
	return workflow.NewContinueAsNewError(ctx, DeployTicketsPortionGCWorkflow, config, nextContinuationToken)
}

func CronDeployTicketsGCWorkflow(ctx workflow.Context, config *Config) error {
	cwo := workflow.ChildWorkflowOptions{
		WorkflowID:        "deployTicketsPortionGC",
		ParentClosePolicy: enums.PARENT_CLOSE_POLICY_ABANDON,
	}
	childCtx := workflow.WithChildOptions(ctx, cwo)
	err := workflow.ExecuteChildWorkflow(childCtx, DeployTicketsPortionGCWorkflow, config, "").GetChildWorkflowExecution().Get(childCtx, nil)
	return err
}
