#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Common classes and functions.
"""

import argparse
import collections
import logging
import os
import sys
import unittest

if sys.hexversion >= 0x03000000:
    from urllib.parse import urljoin
    from xmlrpc.client import ServerProxy
else:
    from urlparse import urljoin
    from xmlrpclib import ServerProxy


class SandboxUrlArgumentType:
    """
    Initializes a server proxy with the specified URL.
    """

    def __call__(self, url):
        return ServerProxy(url, allow_none=True)


class SandboxHostArgumentType:
    """
    Initializes a server proxy with the specified host.
    """

    def __call__(self, host):
        url = urljoin("https://" + host, "sandbox/xmlrpc")
        return ServerProxy(url, allow_none=True)


class Expression:
        """
        Expression parser.
        """

        Operator = collections.namedtuple("Operator", ["priority", "apply"])

        Function = collections.namedtuple("Function", ["arity", "call"])

        OPERATORS = {
            "gt": Operator(priority=4, apply=lambda b, a: a > b),
            "gte": Operator(priority=4, apply=lambda b, a: a >= b),
            "in": Operator(priority=4, apply=lambda b, a: a in b),
            "lt": Operator(priority=4, apply=lambda b, a: a < b),
            "lte": Operator(priority=4, apply=lambda b, a: a <= b),
            "eq": Operator(priority=3, apply=lambda b, a: a == b),
            "neq": Operator(priority=3, apply=lambda b, a: a != b),
            "and": Operator(priority=2, apply=lambda b, a: a and b),
            "or": Operator(priority=1, apply=lambda b, a: a or b),
        }

        FUNCTIONS = {
            "int": Function(arity=1, call=lambda a, **context: int(a)),
            "float": Function(arity=1, call=lambda a, **context: float(a)),
            "len": Function(arity=1, call=lambda a, **context: len(a)),
        }

        def __init__(self, filter_expression, **functions):
            """
            Parses the expression and initializes a new instance of Filter.
            """

            # Initialize functions.
            self._functions = dict()
            self._functions.update(self.FUNCTIONS)
            self._functions.update(functions)
            # Initialize the stack.
            self._postfix, stack = list(), list()
            # Go through the filter.
            for filter_item in filter_expression:
                if filter_item == "[":
                    stack.append(filter_item)
                elif filter_item == "]":
                    while True:
                        if not stack:
                            raise ValueError()
                        stack_item = stack.pop()
                        if stack_item == "[":
                            break
                        self._postfix.append(stack_item)
                    if stack and stack[-1] in self._functions:
                        self._postfix.append(stack.pop())
                elif filter_item in self._functions:
                    stack.append(filter_item)
                elif filter_item in self.OPERATORS:
                    while stack:
                        stack_operator = self.OPERATORS.get(stack[-1])
                        if stack_operator is None or stack_operator.priority < self.OPERATORS[filter_item].priority:
                            break
                        self._postfix.append(stack.pop())
                    stack.append(filter_item)
                else:
                    self._postfix.append(filter_item)
            while stack:
                stack_item = stack.pop()
                if stack_item not in self.OPERATORS:
                    raise ValueError()
                self._postfix.append(stack_item)

        def execute(self, **context):
            """
            Computes the expression.
            """

            stack = list()

            def pop():
                """
                Pops value from the stack.
                """

                item = stack.pop()
                if isinstance(item, str) and item in context:
                    # Get a value from the context.
                    item = context[item]
                # Return parsed value.
                return item

            for item in self._postfix:
                if item in self.OPERATORS:
                    try:
                        a, b = pop(), pop()
                    except IndexError:
                        raise ValueError("Invalid expression at operator '%s'." % item)
                    stack.append(self.OPERATORS[item].apply(a, b))
                elif item in self._functions:
                    function = self._functions[item]
                    if len(stack) < function.arity:
                        raise ValueError("Invalid number of arguments in function '%s', %s expected." % (
                            item, function.arity))
                    arguments = list()
                    for i in range(function.arity):
                        arguments.append(pop())
                    try:
                        # A function is able to use context.
                        result = function.call(*arguments, **context)
                    except Exception as ex:
                        raise ValueError("Error in function call %s%s: %s" % (
                            item, repr(arguments), repr(ex)))
                    stack.append(result)
                else:
                    stack.append(item)

            if len(stack) != 1:
                raise ValueError("Invalid expression result (more than 1 value on the stack): %s" % stack)
            return stack[-1]

        def __repr__(self):
            return "%s[_postfix=%s]" % (Expression.__name__, repr(self._postfix))

        def __nonzero__(self):
            return bool(self._postfix)

        def __bool__(self):
            return self.__nonzero__()


def entry_point(
    main,
    description,
    add_arguments=None,
):
    """
    A tool entry point.
    """

    # Initialize the argument parser.
    argument_parser = argparse.ArgumentParser(
        description=description,
        formatter_class=argparse.RawTextHelpFormatter,
    )
    argument_parser.add_argument(
        "-v",
        "--verbose",
        action="store_true",
        default=False,
        dest="verbose",
        help="log additional info (default: %(default)s)",
    )
    # Add arguments if any.
    if add_arguments is not None:
        add_arguments(argument_parser)
    try:
        # Parse arguments.
        args = argument_parser.parse_args()
    except argparse.ArgumentError:
        sys.exit(os.EX_USAGE)
    # Initialize logging.
    logging.basicConfig(
        format="%(asctime)s %(name)s %(levelname)s: %(message)s",
        level=logging.DEBUG if args.verbose else logging.WARN,
        stream=sys.stderr,
    )
    logger = logging.getLogger(entry_point.__name__)
    # Call the main function.
    if hasattr(args, "sandbox"):
        logger.info("Ping Sandbox ...")
        args.sandbox.ping()
        logger.info("Sandbox is OK.")
    try:
        sys.exit(main(args) or os.EX_OK)
    except KeyboardInterrupt:
        logger.debug("Exit.")


def add_sandbox_arguments(argument_parser):
    """
    Adds the arguments to connect to Sandbox.
    """

    argument_group = argument_parser.add_mutually_exclusive_group()
    argument_group.add_argument(
        "--sandbox-url",
        default="https://sandbox.yandex-team.ru/sandbox/xmlrpc",
        dest="sandbox",
        help="Sandbox XML RPC URL (default: %(default)s)",
        metavar="URL",
        type=SandboxUrlArgumentType(),
    )
    argument_group.add_argument(
        "--sandbox-host",
        dest="sandbox",
        help="Sandbox host",
        metavar="HOST",
        type=SandboxHostArgumentType(),
    )
    argument_parser.add_argument(
        "--page-size",
        default=16,
        dest="page_size",
        help="Sandbox XML RPC request page size (default: %(default)s)",
        metavar="SIZE",
        type=int,
    )


class ExpressionTest(unittest.TestCase):

    def test_execute_true(self):
        context = {"a": True, "b": False, "c": True, "d": False}
        result = Expression("[ a or b ] and c or d".split()).execute(**context)
        self.assertTrue(result)

    def test_execute_false(self):
        context = {"a": False, "b": False, "c": True, "d": False}
        result = Expression("[ a or b ] and c or d".split()).execute(**context)
        self.assertFalse(result)

    def test_execute_function(self):
        result = Expression(
            "get [ get [ a b ] c ]".split(),
            get=Expression.Function(arity=2, call=lambda key, dict_: dict_[key]),
        ).execute(
            a={"b": {"c": 1}},
        )
        self.assertEqual(result, 1)


if __name__ == "__main__":
    unittest.main()
