#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Search for AUTOCHECK_BUILD tasks in Sandbox.

http://eigenein.at.yandex-team.ru/replies.xml?item_no=56
http://eigenein.at.yandex-team.ru/replies.xml?item_no=59
"""

import argparse
import itertools
import logging
import os

import common


def add_arguments(argument_parser):
    common.add_sandbox_arguments(argument_parser)
    argument_parser.add_argument(
        dest="filter_expression",
        help="task filter expression",
        metavar="EXPRESSION",
        nargs=argparse.REMAINDER,
    )
    argument_parser.add_argument(
        "--format",
        default="%(url)s",
        dest="output_format",
        help="output format (default: %(default)s)",
        metavar="FORMAT",
    )
    argument_parser.add_argument(
        "--once",
        action="store_true",
        default=False,
        dest="once",
        help="find one task and exit",
    )
    argument_parser.add_argument(
        "--with-resources",
        action="store_true",
        default=False,
        dest="with_resources",
        help="request task resources info into `resources` variable (slow)",
    )


def dict_get(dict_, key, default_value):
    """
    Gets the key value.
    """

    logger = logging.getLogger(dict_get.__name__)

    value = dict_
    for segment in key.split("."):
        if segment in value:
            value = value[segment]
        else:
            logger.debug("%s=%s", key, default_value)
            return default_value
    logger.debug("%s=%s", key, value)
    return value


def main(args):
    logger = logging.getLogger(main.__name__)

    try:
        task_filter = common.Expression(
            args.filter_expression,
            ctx=common.Expression.Function(
                arity=2,
                call=lambda default_value, key, **context: dict_get(context["task"]["ctx"], key, default_value),
            ),
            get=common.Expression.Function(
                arity=3,
                call=lambda default_value, key, dict_, **context: dict_get(dict_, key, default_value),
            ),
        )
    except ValueError as ex:
        logger.fatal("Invalid expression.")
        return os.EX_USAGE

    logger.debug("Task filter: %s", repr(task_filter))

    for offset in itertools.count(0, args.page_size):
        logger.debug("Requesting tasks at offset %s with page size %s ...", offset, args.page_size)
        tasks = args.sandbox.listTasks({
            "hidden": True,
            "limit": args.page_size,
            "load": True,
            "offset": offset,
            "show_childs": True,
            "task_type": "AUTOCHECK_BUILD",
        })
        if not tasks:
            break
        for task in tasks:
            logger.debug("Task: %s (%s)", task["url"], task["status"])
            if args.with_resources:
                logger.debug("Requesting task resources info ...")
                resources = args.sandbox.listResources({"task_id": task["id"], "omit_failed": False, "hidden": True})
                logger.debug("Found %d resources.", len(resources))
            else:
                resources = None
            try:
                result = not task_filter or task_filter.execute(
                    finished=task["status"] == "FINISHED",
                    resources=resources,
                    task=task,
                )
            except Exception as ex:
                logger.fatal(repr(ex))
                return os.EX_USAGE
            logger.debug("Filter result: %r", result)
            if result:
                print(args.output_format % task)
                if args.once:
                    logger.info("Search stopped (--once).")
                    return


common.entry_point(main, globals()["__doc__"], add_arguments)
