#!/usr/bin/env python

import os
import xmlrpclib

from argparse import ArgumentParser, REMAINDER
from subprocess import Popen, PIPE
from xml.etree import ElementTree as ET

def get_arcadia_root(d):
    r = d
    while r != '/' and not os.path.exists(os.path.join(r, '.arcadia.root')):
        r = os.path.dirname(r)

    return r
    
def get_svn_props(path):
    p = Popen(['svn', 'info', '--xml', '--non-interactive', '--no-auth-cache', '--trust-server-cert'],
              stdout = PIPE, stderr = PIPE, cwd = path)
    (stdout, stderr) = p.communicate()
    assert p.returncode == 0, "'svn info' failed: \n%s" % stderr
    root = ET.fromstring(stdout)
    rev = root.find('entry').attrib['revision']
    svn_url = root.find('entry').find('url').text
    svn_root =  root.find('entry').find('repository').find('root').text
    rp = os.path.relpath(svn_url, svn_root)
    return (rev, rp)

def get_svn_diff(path):
    p = Popen(['svn', 'diff', '--non-interactive', '--no-auth-cache', '--trust-server-cert'],
              stdout = PIPE, stderr = PIPE, cwd = path)
    (stdout, stderr) = p.communicate()
    assert p.returncode == 0, "'svn diff' failed: \n%s" % stderr
    if stdout == '':
        return None
    return stdout


SB_SERVER = 'http://sandbox.yandex-team.ru'
ARCADIA_URL = 'svn+ssh://arcadia.yandex.ru/arc'

if __name__ == "__main__":
    argp = ArgumentParser(description = "Run build in sandbox")
    argp.add_argument('--head', dest = 'build_head', action='store_true', default=False)
    argp.add_argument('--distbs', dest = 'dist', action='store_true', default=False)
    argp.add_argument('--ymake', dest = 'ymake', action='store_true', default=False)
    argp.add_argument('--skip-tests', dest = 'skip_tests', action='store_false', default=True)
    argp.add_argument('--sandbox')
    argp.add_argument('targets', nargs=REMAINDER)
    args = argp.parse_args()
    pwd = os.getcwd()
    arcadia = get_arcadia_root(pwd)
    (rev, branch) = get_svn_props(arcadia)
    
    if args.build_head:
        rev = 'HEAD'

    sb_server = SB_SERVER
    if args.sandbox:
        sb_server = args.sandbox
        
    patch = get_svn_diff(arcadia)

    relpath = os.path.relpath(pwd, arcadia)
    targets = ['all']
    if relpath != ".":
        targets = [relpath]
    if len(args.targets) > 0:
        targets = args.targets
        
    server = xmlrpclib.ServerProxy(sb_server + '/sandbox/xmlrpc', allow_none = True)
    owner = os.getlogin()
    build_tool = 'ninja'
    if args.dist:
        build_tool = 'distbs'

    if args.ymake:
        build_tool = 'ymake'
        
    task_id = server.createTask({'type_name': 'AUTOCHECK_BUILD',
                                 'owner': owner,
                                 'arch': 'linux',
                                 'ctx': {'checkout_arcadia_from_url': '%s/%s@%s' % (ARCADIA_URL, branch, rev),
                                         'autocheck_make_only_dirs': ';'.join(targets),
                                         'arcadia_patch': patch,
                                         'autocheck_build_tool': build_tool,
                                         'autocheck_skip_testing': args.skip_tests,
                                         'notify_via': 'jabber'},
                                 'descr': 'Build rev %s' % str(rev)})

    
    print "Created sandbox task %s/sandbox/tasks/view?task_id=%d" % (sb_server, task_id)

    
