#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Layout renderers.
"""

from __future__ import print_function

import datetime
import logging
import re
import sys


class _Renderer:
    """
    Renderer base class.
    """

    _align_constants = "<>^"

    def render_header(self, max_width):
        """
        Returns the string to be rendered as the column header.
        """

        raise NotImplementedError("render_header")

    def render_value(self, task, max_width):
        """
        Returns the string to be rendered as the column value.
        """

        raise NotImplementedError("render_value")

    def _check_align(self, align):
        if not align in self._align_constants:
            raise ValueError("Invalid align: " + align)

    def _get_format_string(self, width, align):
        """
        Gets the format string for `string.format`.
        """

        return "{{0:{align}{width}}}".format(
            align=align,
            width=width,
        )

    def _shorten_string(self, string, max_width):
        """
        Shortens string down to max_width characters.
        """

        return (
            string if len(string) < max_width
            else string[:max_width - 3] + "..."
        )


class Layout(_Renderer):
    """
    Manages layout renderers.
    """

    _column_re = re.compile(
        r"^(?P<renderer_name>[a-z_]+)(?P<align>[\<\>\:])?(?P<width>\d+)?$",
    )

    def __init__(self, layout_string):
        self._parse_layout(layout_string)

    def _parse_layout(self, layout_string):
        column_strings = layout_string.split()
        self._renderers = []
        for column_string in column_strings:
            column_match = self._column_re.match(column_string)
            if not column_match:
                error_string = "Invalid layout format: " + column_string
                print(error_string, file=sys.stderr)
                raise ValueError(error_string)
            column_layout = column_match.groupdict()
            renderer_name = column_layout["renderer_name"].lower()
            renderer_class = _all_renderers.get(renderer_name)
            if not renderer_class:
                error_string = "Unknown column: " + renderer_name
                print(error_string, file=sys.stderr)
                raise ValueError(error_string)
            column_width, column_align = map(
                column_layout.get,
                ("width", "align"),
            )
            renderer_options = dict()
            if column_width is not None:
                renderer_options["width"] = int(column_width)
            if column_align is not None:
                renderer_options["align"] = column_align
            try:
                renderer = renderer_class(**renderer_options)
            except Exception as ex:
                print(str(ex))
                raise
            self._renderers.append(renderer)

    def _render(self, max_width, render_func):
        """
        Combines the result of render_func on all renderers.
        """

        result = ""
        for renderer in self._renderers:
            if result:
                result = result + " "
            elif len(result) >= max_width:
                break
            result = result + render_func(
                renderer,
                max_width - len(result),
            )
        return result[:max_width]

    def render_header(self, max_width):
        """
        Renders header.
        """

        return self._render(
            max_width,
            lambda renderer, max_width: renderer.render_header(max_width),
        )

    def render_value(self, task, max_width):
        """
        Renders task line.
        """

        return self._render(
            max_width,
            lambda renderer, max_width: renderer.render_value(task, max_width),
        )


class IdRenderer(_Renderer):
    """
    Renders task ID.
    """

    def __init__(self, width=8, align=">"):
        if width < 2:
            raise ValueError("ID column width should be not less than 2.")
        self._check_align(align)

        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("ID")

    def render_value(self, task, max_width):
        return self._format_string.format(task["id"])


class TypeRenderer(_Renderer):
    """
    Renders task type.
    """

    def __init__(self, width=25, align="<"):
        if width < 4:
            raise ValueError("Type column width should be not less than 4.")
        self._check_align(align)

        self._width = width
        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("TYPE")

    def render_value(self, task, max_width):
        return self._format_string.format(self._shorten_string(
            task["type"],
            min(max_width, self._width),
        ))


class PriorityRenderer(_Renderer):
    """
    Renders task priority.
    """

    def __init__(self, width=4, align=">"):
        if width < 4:
            raise ValueError(
                "Priority column width should be not less than 4.",
            )
        self._check_align(align)

        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("PRIO")

    def render_value(self, task, max_width):
        return self._format_string.format(task["prio"])


class HiddenRenderer(_Renderer):
    """
    Renders a hidden flag.
    """

    def __init__(self, width=1, align="<"):
        if width != 1:
            raise ValueError("Hidden column width should be equal to 1.")
        self._check_align(align)

        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("H")

    def render_value(self, task, max_width):
        return self._format_string.format(
            (".", "H")[task["hidden"]],
        )


class DescriptionRenderer(_Renderer):
    """
    Renders task description.
    """

    def __init__(self, width=0, align="<"):
        if width != 0 and width < 13:
            raise ValueError(
                "Description column width should not be less than 11 (or 0)."
            )
        self._width = width
        self._check_align(align)
        self._align = align

    def render_header(self, max_width):
        format_string = self._get_format_string(max_width, self._align)
        return format_string.format("DESCRIPTION")

    def render_value(self, task, max_width):
        format_string = self._get_format_string(max_width, self._align)
        return format_string.format(
            self._shorten_string(task["descr"], max_width),
        )


class StatusRenderer(_Renderer):
    """
    Renders a status flag.
    """

    _status_characters = {
        "EXECUTING": "X",
        "ENQUEUED": "Q",
        "UNKNOWN": "U",
    }

    def __init__(self, width=1, align="<"):
        if width != 1:
            raise ValueError("Status column width should be equal to 1.")
        self._check_align(align)

        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("S")

    def render_value(self, task, max_width):
        return self._format_string.format(
            self._status_characters.get(task["status"], "?"),
        )


class TimeRenderer(_Renderer):
    """
    Renders execution time (or time in queue).
    """

    def __init__(self, width=9, align=">"):
        if width < 9:
            raise ValueError("Time column width should not be less than 9.")
        self._check_align(align)

        self._width = width
        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("TIME")

    def render_value(self, task, max_width):
        timestamp = (
            task["timestamp_start"] if task["status"] == "EXECUTING"
            else task["timestamp"]
        )
        timedelta = datetime.timedelta(
            # Drop milliseconds.
            seconds=int((
                datetime.datetime.utcnow() -
                datetime.datetime.utcfromtimestamp(timestamp)
            ).total_seconds()),
        )
        return self._format_string.format(timedelta)[:self._width]


class ModelRenderer(_Renderer):
    """
    Renders client CPU model.
    """

    def __init__(self, width=8, align=">"):
        if width < 7:
            raise ValueError("Time column width should not be less than 7.")
        self._check_align(align)
        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("MODEL")

    def render_value(self, task, max_width):
        return self._format_string.format(task["model"])


class HostRenderer(_Renderer):
    """
    Renders hostname.
    """

    def __init__(self, width=12, align="<"):
        if width < 7:
            raise ValueError("Time column width should not be less than 7.")
        self._check_align(align)

        self._width = width
        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("HOST")

    def render_value(self, task, max_width):
        return self._format_string.format(task["host"])[:self._width]


class ArchRenderer(_Renderer):

    def __init__(self, width=8, align="<"):
        if width < 5:
            raise ValueError("Arch column width should not be less than 5.")
        self._check_align(align)

        self._width = width
        self._format_string = self._get_format_string(width, align)

    def render_header(self, max_width):
        return self._format_string.format("ARCH")

    def render_value(self, task, max_width):
        return self._format_string.format(task["arch"])[:self._width]


_all_renderers = {
    "id": IdRenderer,
    "type": TypeRenderer,
    "priority": PriorityRenderer,
    "hidden": HiddenRenderer,
    "description": DescriptionRenderer,
    "status": StatusRenderer,
    "time": TimeRenderer,
    "model": ModelRenderer,
    "host": HostRenderer,
    "arch": ArchRenderer,
}
