#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Display Sandbox tasks.
"""

from __future__ import print_function

import argparse
import curses
import datetime
import itertools
import json
import logging
import operator
import os
import socket
import sys
import threading

import sandbox_top.__version__
import sandbox_top.misc


def main(args):
    """
    Main function called with command line arguments.
    """

    if args.show_version_and_exit:
        print(sandbox_top.__version__.version)
        return os.EX_OK
    if args.list_task_types_and_exit:
        print(json.dumps(args.sandbox.listTaskTypes()))
        return os.EX_OK

    logger = logging.getLogger("main")
    logger.info("Running under %s.", sys.version.replace("\n", ""))

    if not args.force_start:
        logger.info("Ping Sandbox ...")
        try:
            args.sandbox.ping()
        except socket.error as ex:
            logger.fatal(str(ex))
            return os.EX_UNAVAILABLE
    logger.info("Entering curses mode ...")
    try:
        curses.wrapper(lambda stdscr: curses_main(args, stdscr))
    except KeyboardInterrupt:
        logger.info("Keyboard interrupt.")
    logger.info("Left curses mode.")


def curses_main(args, stdscr):
    """
    Main function executed in Curses environment.
    """

    stop_event = threading.Event()
    thread = threading.Thread(
        target=refresh_loop,
        name="refresh-thread",
        args=(args, stdscr, stop_event),
    )
    thread.daemon = True

    # Start refreshing thread.
    thread.start()

    # Workaround: make stdscr.getch non-blocking and prevent the loop
    # from eating up to 100% CPU.
    stdscr.timeout(200)

    try:
        while True:
            key = stdscr.getch()
            if key == -1:
                continue
            if key == ord("q"):
                break
    finally:
        stop_event.set()


def refresh_loop(args, stdscr, stop_event):
    """
    Constantly refreshes the screen until stop_event is set.
    """

    # Print initial logo.
    stdscr.addstr(
        0, 0,
        "sandbox-top",
        curses.A_BOLD,
    )
    # Print initial status.
    stdscr.addstr(
        0, len("sandbox-top"),
        " - loading initial screen ...",
    )
    # First refresh.
    refresh_all(args, stdscr)
    # Refresh loop.
    while not stop_event.wait(timeout=args.refresh_interval):
        refresh_all(args, stdscr)


def refresh_all(args, stdscr):
    """
    Refreshes the screen with the newly obtained data.
    """

    is_available, tasks = refresh_data(
        args,
        # Limit returned task list length.
        stdscr.getmaxyx()[0],
    )
    try:
        refresh_screen(stdscr, args.layout, is_available, tasks)
    except curses.error as ex:
        logging.getLogger("refresh_all").warning(str(ex))


def refresh_screen(stdscr, layout, is_available, tasks):
    stdscr.erase()
    stdscr.move(0, 0)
    maxy, maxx = stdscr.getmaxyx()

    # Print summary.
    stdscr.addstr(
        1, 0,
        "Status: Sandbox is{0} available".format("" if is_available else " NOT"),
    )
    stdscr.addstr(
        2, 0, (
            ", ".join("{0}: {1}".format(status, len(tasks)) for status, tasks in tasks.items())
            if is_available else "No tasks information is available."
        ),
    )
    # Clear header.
    stdscr.addstr(
        4, 0,
        " " * maxx,
        curses.A_REVERSE,
    )
    # Print header.
    stdscr.addstr(
        4, 0,
        layout.render_header(maxx),
        curses.A_REVERSE,
    )
    # Merge and sort all tasks by ID descending.
    tasks = sorted(
        itertools.chain(*tasks.values()),
        key=operator.itemgetter("id"),
        reverse=True,
    )
    # Iterate over tasks that fit into screen.
    for task, y in zip(tasks, range(5, maxy - 2)):
        # Print task.
        stdscr.addstr(
            y, 0,
            layout.render_value(task, maxx),
            curses.A_BOLD if task["status"] == "EXECUTING" else curses.A_NORMAL,
        )
    # Print logo.
    stdscr.hline(0, 0, " ", maxx)
    stdscr.addstr(
        0, 0,
        "sandbox-top",
        curses.A_BOLD,
    )
    # Print update datetime.
    stdscr.addstr(
        0, len("sandbox-top"),
        " - {0:%c}".format(datetime.datetime.now()),
    )
    # Print help.
    stdscr.addstr(maxy - 1, 1, "[ ", curses.A_NORMAL)
    stdscr.addstr(maxy - 1, 3, "q", curses.A_BOLD)
    stdscr.addstr(maxy - 1, 4, " - quit ]", curses.A_NORMAL)


def refresh_data(args, limit):
    """
    Obtains Sandbox state.
    """

    try:
        query = {
            "limit": limit,
            "hidden": not args.hide_hidden,
            "load": True,
            "show_childs": args.show_childs,
        }
        # Fill optional parameters.
        if args.task_type:
            query["task_type"] = args.task_type
        if args.owner:
            query["owner"] = args.owner
        # Find tasks.
        status_mapping = {
            "q": "ENQUEUED",
            "x": "EXECUTING",
            "u": "UNKNOWN",
        }
        tasks = dict()
        for status_char in args.statuses:
            status = status_mapping[status_char.lower()]
            query["status"] = status
            tasks[status] = args.sandbox.listTasks(query)
    except Exception as ex:
        # Prevent spoiling the screen.
        if args.log_file is not sys.stderr:
            logging.getLogger("refresh_data").warning(str(ex))
        return False, dict()
    else:
        return True, tasks


def entry_point():
    """
    Package entry point.
    """

    try:
        args = sandbox_top.misc.create_argument_parser().parse_args()
    except argparse.ArgumentError as ex:
        print(str(ex), file=sys.stderr)
        sys.exit(os.EX_USAGE)
    else:
        logging.basicConfig(
            format="%(asctime)s [%(process)d] %(name)s %(levelname)s: %(message)s",
            level=logging.INFO,
            stream=args.log_file,
        )
        sys.exit(main(args) or os.EX_OK)
