#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Miscellaneous functions.
"""

import argparse
import sys

if sys.hexversion >= 0x03000000:
    from urllib.parse import urljoin
    from xmlrpc.client import ServerProxy
else:
    from urlparse import urljoin
    from xmlrpclib import ServerProxy

import sandbox_top.layouts


def create_argument_parser():
    """
    Initializes argument parser.
    """

    parser = argparse.ArgumentParser(
        description="Display Sandbox tasks.",
        formatter_class=argparse.RawTextHelpFormatter,
        prog="sandbox-top",
    )

    # General.
    parser.add_argument(
        "--log-path",
        metavar="LOG_PATH",
        help="log file path",
        default=sys.stderr,
        dest="log_file",
        type=argparse.FileType("wt"),
    )
    # Sandbox location.
    url_host_group = parser.add_mutually_exclusive_group()
    url_host_group.add_argument(
        "--xml-rpc-url",
        metavar="URL",
        help="Sandbox XML RPC URL (default: %(default)s)",
        default="https://sandbox.yandex-team.ru/sandbox/xmlrpc",
        dest="sandbox",
        type=ServerProxy,
    )
    url_host_group.add_argument(
        "--host",
        metavar="HOST",
        help="Sandbox host (optionally followed by :port)",
        default=None,
        dest="sandbox",
        type=lambda host: ServerProxy(
            urljoin("http://" + host, "sandbox/xmlrpc"),
        ),
    )
    # Filtering.
    parser.add_argument(
        "--show-childs",
        help="show subtasks",
        default=False,
        dest="show_childs",
        action="store_true",
    )
    parser.add_argument(
        "--hide-hidden",
        help="do not show hidden tasks",
        default=False,
        dest="hide_hidden",
        action="store_true",
    )
    parser.add_argument(
        "--task-type",
        metavar="TASK_TYPE",
        help="task type",
        default=None,
        dest="task_type",
    )
    parser.add_argument(
        "--owner",
        metavar="USERNAME",
        help="task owner",
        default=None,
        dest="owner",
    )
    parser.add_argument(
        "--statuses",
        metavar="STATUS",
        help="task status",
        default=["q", "x", "u"],
        dest="statuses",
        choices=["q", "Q", "x", "X","u", "U"],
        nargs="+",
    )
    # Screen.
    parser.add_argument(
        "--interval",
        metavar="SECONDS",
        help="refresh interval (default: %(default)s)",
        default=5.0,
        dest="refresh_interval",
        type=float,
    )
    parser.add_argument(
        "--layout",
        metavar="LAYOUT_STRING",
        help="specifies the rendering layout (default: %(default)s)",
        default="id>8 arch<5 type<25 priority>4 hidden<1 status<1 time>9 description<0",
        dest="layout",
        type=sandbox_top.layouts.Layout,
    )
    # Other.
    parser.add_argument(
        "--force-start",
        help="do not ping Sandbox at start",
        default=False,
        action="store_true",
        dest="force_start",
    )
    special_option_group = parser.add_mutually_exclusive_group()
    special_option_group.add_argument(
        "--version",
        help="show version and exit",
        default=False,
        action="store_true",
        dest="show_version_and_exit",
    )
    special_option_group.add_argument(
        "--list-task-types",
        help="show available task types and exit",
        default=False,
        action="store_true",
        dest="list_task_types_and_exit",
    )

    return parser
