#include "service.h"

#include <infra/udp_click_metrics/api/api.pb.h>

#include <infra/libs/service_iface/request.h>
#include <infra/libs/service_iface/str_iface.h>

#include <search/begemot/rules/init/util/util.h>

#include <quality/logs/baobab/api/cpp/common/event_impl.h>
#include <quality/logs/baobab/api/cpp/common/events_parser.h>
#include <quality/logs/baobab/api/cpp/common/parser_config.h>
#include <quality/logs/baobab/api/cpp/common/visitor.h>
#include <quality/logs/baobab/api/cpp/transport_context/event_context.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/uri/uri.h>

#include <util/generic/serialized_enum.h>
#include <util/generic/mapfindptr.h>
#include <util/string/split.h>

namespace NUdpClickMetrics {

namespace {
constexpr ui32 BUCKETS_NUMBER = 5;

void ParseSlots(const TStringBuf slots, TVector<std::pair<ui32, i32>>& result) {
    for (TStringBuf slot : StringSplitter(slots).Split(';').SkipEmpty()) {
        ui32 testId;
        TStringBuf slotId;
        i32 bucket;
        if (StringSplitter(slot).Split(',').TryCollectInto(&testId, &slotId, &bucket) && bucket != -1) {
            result.push_back({testId, bucket});
        }
    }
}

TMaybe<TString> GetWizardName(const NJson::TJsonValue& clickEvent) {
    if (clickEvent["fast"]["organic"] == 1) {
        return "organic";
    } else if (TString wizardName = clickEvent["fast"]["wzrd"].GetString(); !wizardName.empty()) {
        if (wizardName == "health" || wizardName == "images" || wizardName.size() != 6) {
            return wizardName;
        }
    }

    return Nothing();
}

void HandleClickEvent(const NJson::TJsonValue& clickEvent, NUdpMetrics::NApi::TMetric& request) {
    request.SetName("click");

    if (TString service = clickEvent["service"].GetString(); !service.empty()) {
        (*request.MutableLabels())["service_name"].AddValues()->SetValue(std::move(service));
    }
    (*request.MutableLabels())["subservice"].AddValues()->SetValue(clickEvent["subservice"].GetString());

    if (TMaybe<TString> wizardName = GetWizardName(clickEvent); wizardName.Defined()) {
        (*request.MutableLabels())["wizard"].AddValues()->SetValue(std::move(wizardName.GetRef()));
    }
}

void ParseBaobabEventJson(const TStringBuf eventString, NUdpMetrics::NApi::TMetric& request) {
    const TString baobabEvent = UrlUnescapeRet(eventString);
    if (NJson::TJsonValue baobabEventJson; NJson::ReadJsonFastTree(baobabEvent, &baobabEventJson)) {
        if (baobabEventJson.IsArray()) {
            for (const NJson::TJsonValue& event : baobabEventJson.GetArraySafe()) {
                if (event["event"] == "click") {
                    HandleClickEvent(event, request);
                    return;
                }
            }
        } else if (baobabEventJson.IsMap() && baobabEventJson["event"] == "click") {
            HandleClickEvent(baobabEventJson, request);
        }
    }
}

bool ParseVars(const TStringBuf vars, NUdpMetrics::NApi::TMetric& request) {
    for (const TStringBuf var : StringSplitter(vars).Split(',').SkipEmpty()) {
        TStringBuf key;
        TStringBuf value;
        var.Split('=', key, value);

        if (key == "-baobab-event-json") {
            ParseBaobabEventJson(value, request);
            return true;
        }
    }
    return false;
}

void ParseEvents(const TStringBuf events, NUdpMetrics::NApi::TMetric& request) {
    ParseBaobabEventJson(events, request);
}

const static THashMap<TString, EDeviceType> DEVICE_BY_PATH = {
    {"yandsearch",    EDeviceType::DESKTOP},
    {"search/family", EDeviceType::DESKTOP},
    {"search",        EDeviceType::DESKTOP},
    {"padsearch",     EDeviceType::TABLET},
    {"search/pad",    EDeviceType::TABLET},
    {"touchsearch",   EDeviceType::TOUCH},
    {"search/touch",  EDeviceType::TOUCH},
    {"msearch",       EDeviceType::SMART},
    {"search/smart",  EDeviceType::SMART}
};

EDeviceType GetDeviceType(const uatraits::detector& deviceDetector, const TStringBuf userAgent, const TStringBuf path, const TQuickCgiParam& cgi) {
    if (auto it = cgi.find("ui"); it != cgi.end() && it->second == "webmobileapp.yandex" && path.StartsWith("search/touch")) {
        return EDeviceType::TOUCH;
    }

    if (auto it = cgi.find("pad_to_touch"); it != cgi.end() && !it->second.empty() && it->second != "0") {
        return EDeviceType::TOUCH;
    }

    if (auto it = cgi.find("padp"); it != cgi.end()) {
        if (it->second == "touch") {
            return EDeviceType::TOUCH;
        } else if (it->second == "desktop") {
            return EDeviceType::DESKTOP;
        }
    }

    if (const EDeviceType* deviceType = DEVICE_BY_PATH.FindPtr(path)) {
        return *deviceType;
    }

    uatraits::detector::result_type device = deviceDetector.detect(userAgent.data());

    const TStringBuf browserName = device["BrowserName"];
    const TString browserVersion = NInit::CastVersion(device["BrowserVersion"]);
    const TStringBuf browserEngine = device["BrowserEngine"];
    const TString browserEngineVersion = NInit::CastVersion(device["BrowserEngineVersion"]);
    const TStringBuf osFamily = device["OSFamily"];
    const TString osVersion = NInit::CastVersion(device["OSVersion"]);

    if (device["isTv"] == "true") {
        return EDeviceType::DESKTOP;
    } else if (device["isTablet"] == "true") {
        if (browserName == "OperaMini") {
            return EDeviceType::DESKTOP;
        } else if ((osFamily == "iOS" && osVersion >= NInit::CastVersion("6")) || (osFamily == "Android" && osVersion >= NInit::CastVersion("4"))) {
            return EDeviceType::TABLET;
        } else {
            return EDeviceType::DESKTOP;
        }
    } else if (device["isMobile"] == "true") {
        if (browserName == "OperaMini" && browserEngine == "WebKit") {
            return EDeviceType::TOUCH;
        } else if (browserName == "OperaMini" || browserName == "MobileFirefox" || osFamily == "Symbian") {
            return EDeviceType::SMART;
        } else if (browserName == "OperaMobile") {
            if (browserVersion >= NInit::CastVersion("12")) {
                return EDeviceType::TOUCH;
            } else {
                return EDeviceType::SMART;
            }
        } else if ((osFamily == "iOS" && osVersion >= NInit::CastVersion("4")) ||
                   (osFamily == "Android" && osVersion >= NInit::CastVersion("2.1")) ||
                   (osFamily == "WindowsPhone" && osVersion >= NInit::CastVersion("7.5")) ||
                   (osFamily == "Tizen" && osVersion >= NInit::CastVersion("2")))
        {
            return EDeviceType::TOUCH;
        } else {
            return EDeviceType::SMART;
        }
    } else {
        if ((browserName == "MSIE" && (browserVersion < NInit::CastVersion("8") || (browserVersion >= NInit::CastVersion("8") && browserVersion <= NInit::CastVersion("9"))) ||
            (browserEngine == "Gecko" && browserEngineVersion < NInit::CastVersion("23"))))
        {
            return EDeviceType::DESKTOP;
        } else {
            return EDeviceType::DESKTOP;
        }
    }
}

void ParseDevice(const uatraits::detector& deviceDetector, const TStringBuf userAgent, const TStringBuf referer, NUdpMetrics::NApi::TMetric& request) {
    NUri::TUri uriParser;
    uriParser.Parse(referer, NUri::TFeature::FeaturesRecommended | NUri::TFeature::FeaturesCheckSpecialChar);

    TStringBuf path = uriParser.GetField(NUri::TField::FieldPath);
    path.SkipPrefix("/");

    const TQuickCgiParam cgi(uriParser.GetField(NUri::TField::FieldQuery));

    EDeviceType deviceType = GetDeviceType(deviceDetector, userAgent, path, cgi);
    (*request.MutableLabels())["device"].AddValues()->SetValue(ToString(deviceType));
}

static const THashSet<TString> NOT_CLICK_PATH_PREFIXES = {{
    "690",
}};

bool IsClickPath(const TStringBuf path) {
    return !path.empty() && !NOT_CLICK_PATH_PREFIXES.contains(path.Before('.'));
}

TString GetMordaMonitoringMetricsPath(TStringBuf path) {
    const auto& left = path.NextTok(".");
    const auto& right = path.NextTok(".");
    return TString::Join(left, ".", right);
}

TString BuildMordaMonitoringMetricsPath(const NJson::TJsonValue& event) {
    const TString& parentPath = event["parent-path"].GetString();
    if (parentPath.empty()) {
        return TString();
    }
    const auto& blocks = event["blocks"].GetArray();
    if (blocks.empty()) {
        return parentPath;
    }
    const TString& ctag = blocks[0]["ctag"].GetString();
    return TString::Join(parentPath, ".", ctag);
}

TString GetMordaMonitoringMetricsDevice(TStringBuf path) {
    return TString{path.NextTok(".")};
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleMordaMonitoringMetrics(const THashMap<TStringBuf, TStringBuf>& tokens, NUdpMetrics::NApi::TMetric metric) {
    NUdpMetrics::NApi::TReqIncreaseMetrics apiRequest;

    if (const TStringBuf* lidToken = tokens.FindPtr("lid")) {
        metric.SetName("click"); // click ?
        (*metric.MutableLabels())["path"].AddValues()->SetValue(GetMordaMonitoringMetricsPath(*lidToken));
        (*metric.MutableLabels())["device"].AddValues()->SetValue(GetMordaMonitoringMetricsDevice(*lidToken));
        *apiRequest.AddMetrics() = std::move(metric);
        return nullptr; // temporarily turn off GET queries
    } else if (const TStringBuf* eventsToken = tokens.FindPtr("events")) {
        const TString& events = UrlUnescapeRet(*eventsToken);
        NJson::TJsonValue eventsJson;
        NJson::ReadJsonFastTree(events, &eventsJson);

        for (const auto& event: eventsJson.GetArray()) {
            if (const TString& eventName = event["event"].GetString(); eventName == "click") {
                const TString& path = BuildMordaMonitoringMetricsPath(event); // do not consider case with no empty blocks ?
                if (path.empty()) {
                    continue;
                }
                NUdpMetrics::NApi::TMetric metricForEvent = metric;
                metricForEvent.SetName(eventName);
                const TString& device = GetMordaMonitoringMetricsDevice(path);
                (*metricForEvent.MutableLabels())["device"].AddValues()->SetValue(device);
                (*metricForEvent.MutableLabels())["path"].AddValues()->SetValue(path.substr(device.size() + 1));
                (*metricForEvent.MutableLabels())["geo"].AddValues()->MutableNotSet();
                (*metricForEvent.MutableLabels())["contour"].AddValues()->MutableNotSet();
                if (const TStringBuf* geoToken = tokens.FindPtr("geo")) {
                    if (const TStringBuf* contourToken = tokens.FindPtr("contour")) {
                        (*metricForEvent.MutableLabels())["geo"].AddValues()->SetValue(TString{*geoToken});
                        (*metricForEvent.MutableLabels())["contour"].AddValues()->SetValue(TString{*contourToken});
                    }
                }
                *apiRequest.AddMetrics() = std::move(metricForEvent);
            } // else if (eventName == "show")
        }
    }

    return NInfra::RequestPtr<NInfra::TProtoRequest<NUdpMetrics::NApi::TReqIncreaseMetrics>>("udp", std::move(apiRequest), NInfra::TAttributes{});
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleGeneralOriginalClickdaemonMetrics(
        const TString& userAgent,
        const TString& referer,
        NUdpMetrics::NApi::TMetric metric,
        const uatraits::detector* deviceDetector,
        const THashMap<TStringBuf, TStringBuf>& tokens)
{
    bool hasBaobabEvents = false;
    TVector<std::pair<ui32, i32>> slots;

    if (const TStringBuf* slotsToken = tokens.FindPtr("slots")) {
        ParseSlots(*slotsToken, slots);
    }

    if (const TStringBuf* eventsToken = tokens.FindPtr("events")) {
        ParseEvents(*eventsToken, metric);
        hasBaobabEvents = true;
    } else if (const TStringBuf* varsToken = tokens.FindPtr("vars")) {
        hasBaobabEvents = ParseVars(*varsToken, metric);
    }

    if (metric.GetName().empty()) {
        const TStringBuf* dtypeToken = tokens.FindPtr("dtype");
        const TStringBuf* pathToken = tokens.FindPtr("path");
        if (!hasBaobabEvents && (dtypeToken && *dtypeToken == "iweb") && (pathToken && IsClickPath(*pathToken))) {
            metric.SetName("click");
        } else {
            return nullptr;
        }
    } else if (metric.GetLabels().at("service_name").GetValues().rbegin()->GetValue() != "web") {
        return nullptr;
    }

    if (const TStringBuf* geoToken = tokens.FindPtr("geo")) {
        if (const TStringBuf* contourToken = tokens.FindPtr("contour")) {
            (*metric.MutableLabels())["geo"].AddValues()->SetValue(TString{*geoToken});
            (*metric.MutableLabels())["contour"].AddValues()->SetValue(TString{*contourToken});
        }
    }

    if (metric.GetLabels().at("wizard").GetValues().size() > 1) {
        ParseDevice(*deviceDetector, userAgent, referer, metric);
    }

    NUdpMetrics::NApi::TReqIncreaseMetrics apiRequest;

    for (const auto& [testId, bucket] : slots) {
        NUdpMetrics::NApi::TMetric metricWithTestid = metric;
        (*metricWithTestid.MutableLabels())["testid"].AddValues()->SetValue(ToString(testId));
        (*metricWithTestid.MutableLabels())["bucket"].AddValues()->SetValue(ToString(bucket % BUCKETS_NUMBER));
        *apiRequest.AddMetrics() = std::move(metricWithTestid);
    }

    (*metric.MutableLabels())["testid"].AddValues()->MutableNotSet();
    *apiRequest.AddMetrics() = std::move(metric);

    return NInfra::RequestPtr<NInfra::TProtoRequest<NUdpMetrics::NApi::TReqIncreaseMetrics>>("udp", std::move(apiRequest), NInfra::TAttributes{});
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleOriginalClickdaemonMetrics(
        const TString& redirLogPart,
        const TString& userAgent,
        const TString& referer,
        NUdpMetrics::NApi::TMetric metric,
        const uatraits::detector* deviceDetector)
{
    THashMap<TStringBuf, TStringBuf> tokens;

    for (const TStringBuf token : StringSplitter(redirLogPart).SplitByString("@@").SkipEmpty()) {
        TStringBuf key;
        TStringBuf value;
        token.Split('=', key, value);
        tokens.insert({key, value});
    }

    if (TStringBuf* dtypeToken = tokens.FindPtr("dtype"); dtypeToken && (*dtypeToken == "clck")) {
        return HandleMordaMonitoringMetrics(tokens, {});
        // we are cutting off all clck events here
    }

    return HandleGeneralOriginalClickdaemonMetrics(userAgent, referer, metric, deviceDetector, tokens);
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleQuasarMetrics(const NProtoLogger::NApi::TReqWriteMetrics2LogQuasar& quasarMetrics, NUdpMetrics::NApi::TMetric metric) {
    const TString metricName = quasarMetrics.GetMetricName();
    if (metricName.empty()) {
        return nullptr;
    }

    NUdpMetrics::NApi::TReqIncreaseMetrics apiRequest;

    metric.SetName(metricName);
    (*metric.MutableLabels())["Platform"].AddValues()->SetValue(quasarMetrics.GetPlatform());
    (*metric.MutableLabels())["QuasmodromGroup"].AddValues()->SetValue(quasarMetrics.GetQuasmodromGroup());
    (*metric.MutableLabels())["SoftwareVersion"].AddValues()->SetValue(quasarMetrics.GetSoftwareVersion());
    (*metric.MutableLabels())["QuasmodromSubgroup"].AddValues()->SetValue(quasarMetrics.GetQuasmodromSubgroup());


    *apiRequest.AddMetrics() = std::move(metric);

    return NInfra::RequestPtr<NInfra::TProtoRequest<NUdpMetrics::NApi::TReqIncreaseMetrics>>("udp", std::move(apiRequest), NInfra::TAttributes{});
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleSearchappMordaMetrics(const NProtoLogger::NApi::TReqCardShownLog& searchappMordaMetrics, NUdpMetrics::NApi::TMetric metric) {
    metric.SetName("event");

    NUdpMetrics::NApi::TReqIncreaseMetrics apiRequest;

    (*metric.MutableLabels())["CardId"].AddValues()->SetValue(searchappMordaMetrics.GetCardId());
    (*metric.MutableLabels())["AppVersion"].AddValues()->SetValue(searchappMordaMetrics.GetAppVersion());
    (*metric.MutableLabels())["AppPlatform"].AddValues()->SetValue(searchappMordaMetrics.GetAppPlatform());
    (*metric.MutableLabels())["DeviceModel"].AddValues()->SetValue(searchappMordaMetrics.GetDeviceModel());
    (*metric.MutableLabels())["OsVersion"].AddValues()->SetValue(searchappMordaMetrics.GetOsVersion());

    for (const auto& testId: searchappMordaMetrics.GetTestIds()) {
        (*metric.MutableLabels())["TestId"].AddValues()->SetValue(testId);
    }

    *apiRequest.AddMetrics() = std::move(metric);

    return NInfra::RequestPtr<NInfra::TProtoRequest<NUdpMetrics::NApi::TReqIncreaseMetrics>>("udp", std::move(apiRequest), NInfra::TAttributes{});
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleMegamindIntentsMetrics(const NApi::TMegamindIntentMetrics& mmIntentsMetrics, NUdpMetrics::NApi::TMetric metric) {
    metric.SetName("megamind_intents");

    NUdpMetrics::NApi::TReqIncreaseMetrics apiRequest;

    (*metric.MutableLabels())["Intent"].AddValues()->SetValue(mmIntentsMetrics.GetIntent());
    (*metric.MutableLabels())["ScenarioName"].AddValues()->SetValue(mmIntentsMetrics.GetScenarioName());
    (*metric.MutableLabels())["IsWinnerScenario"].AddValues()->SetValue(mmIntentsMetrics.GetIsWinnerScenario());

    *apiRequest.AddMetrics() = std::move(metric);

    return NInfra::RequestPtr<NInfra::TProtoRequest<NUdpMetrics::NApi::TReqIncreaseMetrics>>("udp", std::move(apiRequest), NInfra::TAttributes{});
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> HandleMegamindIrrelevantMetrics(const NApi::TMegamindIrrelevantMetrics& mmIrrelevantMetrics, NUdpMetrics::NApi::TMetric metric) {
    metric.SetName("megamind_irrelevant");

    NUdpMetrics::NApi::TReqIncreaseMetrics apiRequest;

    (*metric.MutableLabels())["IntentName"].AddValues()->SetValue(mmIrrelevantMetrics.GetIntentName());
    (*metric.MutableLabels())["ScenarioName"].AddValues()->SetValue(mmIrrelevantMetrics.GetScenarioName());

    *apiRequest.AddMetrics() = std::move(metric);

    return NInfra::RequestPtr<NInfra::TProtoRequest<NUdpMetrics::NApi::TReqIncreaseMetrics>>("udp", std::move(apiRequest), NInfra::TAttributes{});
}

} // anonymous namespace


TService::TService(const NUdpMetrics::TUdpMetricsServiceConfig& config)
    : NUdpMetrics::TService(config)
{
    (*BaseMetric_.MutableLabels())["wizard"].AddValues()->MutableNotSet();
    (*BaseMetric_.MutableLabels())["device"].AddValues()->MutableNotSet();
    (*BaseMetric_.MutableLabels())["service_name"].AddValues()->MutableNotSet();
    (*BaseMetric_.MutableLabels())["subservice"].AddValues()->MutableNotSet();
    (*BaseMetric_.MutableLabels())["bucket"].AddValues()->MutableNotSet();
    (*BaseMetric_.MutableLabels())["geo"].AddValues()->MutableNotSet();
    (*BaseMetric_.MutableLabels())["contour"].AddValues()->MutableNotSet();

    const TString browser = NResource::Find("browser.xml");
    const TString profiles = NResource::Find("profiles.xml");
    DeviceDetector_.Reset(MakeHolder<const uatraits::detector>(browser.data(), browser.size(), profiles.data(), profiles.size()));
}

NInfra::TRequestPtr<NUdpMetrics::NApi::TReqIncreaseMetrics> TService::ParseUdpPacket(NUdp::TUdpPacket packet, NInfra::TLogFramePtr /* logFrame */) const {
    NApi::TReqIncreaseClickMetrics request;
    Y_PROTOBUF_SUPPRESS_NODISCARD request.ParseFromString(packet.Data);

    if (request.HasQuasarMetrics()) {
        return HandleQuasarMetrics(request.GetQuasarMetrics(), {});
    } if (request.HasSearchappMordaMetrics()) {
        return HandleSearchappMordaMetrics(request.GetSearchappMordaMetrics(), {});
    } if (request.HasMegamindIntentMetrics()) {
        return HandleMegamindIntentsMetrics(request.GetMegamindIntentMetrics(), {});
    } if (request.HasMegamindIrrelevantMetrics()) {
        return HandleMegamindIrrelevantMetrics(request.GetMegamindIrrelevantMetrics(), {});
    } else {
        TString redirLogPart = request.HasClickdaemonMetrics() ? request.GetClickdaemonMetrics().GetRedirLogPart() : request.GetRedirLogPart();
        TString userAgent = request.HasClickdaemonMetrics() ? request.GetClickdaemonMetrics().GetUserAgent() : request.GetUserAgent();
        TString referer = request.HasClickdaemonMetrics() ? request.GetClickdaemonMetrics().GetReferer() : request.GetReferer();
        return HandleOriginalClickdaemonMetrics(redirLogPart, userAgent, referer, BaseMetric_, DeviceDetector_.Get());
    }
}

} // namespace NUdpClickMetrics
