import { readFileSync } from 'fs';

import { releaseArguments } from './readArgs';
import { SandboxResource, sbApiGetResource } from './sandboxApi';
import { prettyLog, waitPromise } from './utils';
import { ReleaseResource, ypApiCreateObject } from './ypApi';

const { title, resourceId, resourceType, targets, releaseType } = releaseArguments;

prettyLog(releaseArguments);

const SANDBOX_TOKEN = process.env.SANDBOX_OAUTH_TOKEN!;
const PROJECT_NAME = process.env.PROJECT_NAME!;
const RELEASE_AUTHOR = process.env.RELEASE_AUTHOR!;

const NANNY_SERVICE_ID = process.env.NANNY_SERVICE_ID!;

async function createDeployRelease(resource: SandboxResource) {
   const attributes: Record<string, string> = {
      key: resource.attributes.key as string,
      project: resource.attributes.project as string,
   };

   const releaseResource: Partial<ReleaseResource> = {
      arch: 'any',
      attributes: attributes,
      description: resource.description,
      file_md5: resource.md5,
      filename: resource.file_name,
      resource_id: resource.id.toString(),
      skynet_id: resource.skynet_id,
      type: resource.type,
   };

   const release = {
      meta: { id: `${PROJECT_NAME}_${resourceId}` },
      spec: {
         sandbox: {
            release_author: RELEASE_AUTHOR,
            release_type: releaseType,
            resources: [releaseResource],
            task_author: RELEASE_AUTHOR,
            task_id: resource.task.id.toString(),
            task_type: 'TRENDBOX_CI_JOB_BETA',
            title,
         },
         title,
      },
   };

   console.log('Deploy release');
   prettyLog(release);

   await ypApiCreateObject('release', release);
}

async function createNannyRelease() {
   const { createRelease } = require('@yandex-int/trendbox-ci.nanny-workflow');

   const version: string = JSON.parse(readFileSync('../ui/package.json', { encoding: 'utf8' })).version;

   const release = {
      title: `Release version ${version}`,
      resourceId,
      resourceType,
      releaseType,
      ensureDeploy: false,

      nannyServiceId: NANNY_SERVICE_ID,
      sandboxOwner: 'INFRACLOUDUI',
      sandboxToken: SANDBOX_TOKEN,
   };

   console.log('Nanny release');
   prettyLog(release);

   await createRelease(release);
}

const targetHandlers: Record<string, (r: SandboxResource) => Promise<void>> = {
   deploy: createDeployRelease,
   nanny: createNannyRelease,
};

async function run() {
   const resource = await sbApiGetResource(resourceId);

   // run parallel
   await Promise.all(targets.map(t => targetHandlers[t](resource) ?? Promise.resolve()));

   return 'Done';
}

waitPromise(run);
