import { StagePage } from '../../../page_objects/stage/StagePage';
import { getStageName, ROUTES, TIMEOUTS } from '../../../utils';

context('DeployTicketsLoadMore', () => {
   const stageName = getStageName('deploy_tickets_list');
   const page = new StagePage(stageName);
   const tab = page.deployTicketsTab;

   before(() => {
      cy.loginRobot();
      cy.emptyPage();

      // sure stage isn't exists
      cy.deleteStage(stageName);
      cy.createStageWithStatic(stageName);
   });

   after(() => {
      cy.deleteStage(stageName);
   });

   const createApprovalPolicyUI = () => {
      cy.get(tab.buttons.approvalPolicy).click();
      tab.approvalPolicy.fillSimpleApprovalPolicy();
   };

   const createDraftDeployTicketUI = () => {
      cy.get(page.buttons.editLink, { timeout: TIMEOUTS.superSlow }).click();
      page.editTab.createDraftDeployTicket();
   };

   const createDraftDeployTickets = (countOfTickets: number) => {
      cy.pushUrl(ROUTES.stageDeployTickets(stageName));
      cy.location('pathname').should('eq', tab.pathname);

      createApprovalPolicyUI();

      for (let i = 0; i < countOfTickets; i += 1) {
         createDraftDeployTicketUI();
      }
   };

   it('loadMore Tickets', () => {
      let initialTicketsPart;
      const limit = 3;

      createDraftDeployTickets(limit * 2 + 1);

      cy.pushUrl(`${ROUTES.stageDeployTickets(stageName)}?limitInTest=${limit}`);
      cy.reload(); // reset state

      const forceRefreshButton = cy.get(`${tab.buttons.forceRefresh} button`);
      forceRefreshButton.should('exist');
      cy.get(tab.loadMoreLink, { timeout: TIMEOUTS.fast }).should('exist');
      forceRefreshButton.should('not.be.disabled');

      const getTicketsTimestamps = ticketsDatesElements =>
         ticketsDatesElements.toArray().map(i => Number(i.dataset['creationDate']));
      const checkTicketsDatesOrder = ticketsDates => {
         const sortedTicketsDates = [...ticketsDates].sort((a, b) => b - a);
         expect(ticketsDates).to.deep.eq(sortedTicketsDates, 'элементы выводятся в порядке "новые-первыми"');
      };

      cy.get(tab.ticketsList)
         .find(tab.row)
         .should(ticketsDatesElements => {
            expect(ticketsDatesElements).to.have.length(limit);
            const ticketsDates = getTicketsTimestamps(ticketsDatesElements);
            checkTicketsDatesOrder(ticketsDates);
            initialTicketsPart = ticketsDates;
         });

      cy.get(tab.loadMoreLink).click();
      cy.wait(TIMEOUTS.slow);
      cy.get(`${tab.buttons.forceRefresh} button`).should('not.be.disabled');

      cy.get(tab.ticketsList)
         .find(tab.row)
         .should(ticketsDatesElements => {
            expect(ticketsDatesElements).to.have.length(limit * 2);
            const ticketsDates = getTicketsTimestamps(ticketsDatesElements);
            checkTicketsDatesOrder(ticketsDates);
            // проверка соответствия порядка бекенда и клиента
            expect(ticketsDates.slice(0, limit)).to.deep.eq(
               initialTicketsPart,
               'подгрузка новых тикетов не поменяла порядок элементов первой порции',
            );
         });
   });
});
