import { StageEditPage } from 'page_objects/stage/StageEditPage';
import { StageHistoryPage } from 'page_objects/stage/StageHistoryPage';
import {
   EMPTY_VALUE,
   getStageName,
   ROUTES,
   STAGE_LOCATIONS,
   DEFAULT_DEPLOY_UNIT,
   DEFAULT_BOX,
   DEFAULT_WORKLOAD,
} from 'utils';

context('StageHistoryRollback', () => {
   const duId = DEFAULT_DEPLOY_UNIT;
   const boxId = DEFAULT_BOX;
   const workloadId = DEFAULT_WORKLOAD;

   const stageName = getStageName('history-rollback');
   const stageHistoryPage = new StageHistoryPage(stageName);
   const stageEditPage = new StageEditPage(stageName);
   const duEditForm = stageEditPage.hugeForm.deployUnit(duId);

   before(() => {
      cy.loginRobot();
      cy.emptyPage();

      // sure stage isn't exists
      cy.deleteStage(stageName);
   });

   after(() => {
      cy.deleteStage(stageName);
   });

   const countClusters = STAGE_LOCATIONS.length;

   const message = {
      revision2: 'testcafe update stage',
      revision3: 'apply revision 1 as is',
      revision4: 'apply revision 2 with changes',
   };

   const podCount = {
      revision1: 1,
      revision2: 2,
   };

   const disruptionBudget = {
      revision1: countClusters * podCount.revision1 - 1,
      revision2: countClusters * podCount.revision2 - 1,
   };

   it('History rollback', () => {
      /******************************************
       * подготовка двух первых ревизий *
       ******************************************/
      cy.log('# подготовка двух первых ревизий');
      cy.createSimpleStage(stageName, {revisionMessage: ''});

      stageEditPage.visit();

      // by @nodejsgirl
      // TODO: check values (testcafe)
      // TODO: check edited values after "Edit" (testcafe)
      // TODO: check edited values after "Cancel" (testcafe)


      cy.get(duEditForm.treeItem.linkSelector).click();
      duEditForm.form.set({
         locations: STAGE_LOCATIONS.map(id => ({ id, podCount: podCount.revision2 })),
         disruptionBudget: String(disruptionBudget.revision2),
      });

      stageEditPage.buttons.update.click();

      stageEditPage.diffView.deploy(message.revision2);

      /******************************************
       * проверка истории двух первых ревизий *
       ******************************************/
      cy.log('# проверка истории двух первых ревизий');
      stageHistoryPage.visit();
      const revisionRow1 = stageHistoryPage.getRevisionRow(1);
      const revisionRow2 = stageHistoryPage.getRevisionRow(2);

      cy.log('сообщения в списке');
      revisionRow2.currentRevision.should('exist');
      revisionRow2.message(message.revision2).should('exist');
      revisionRow1.message(EMPTY_VALUE).should('exist');

      cy.log('переходы между страницами');
      revisionRow1.row.click();

      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '1'));
      stageHistoryPage.revision.empty.should('exist');

      stageHistoryPage.buttons.backToHistory.click();
      revisionRow2.row.click();

      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '2'));

      stageHistoryPage.revision.text(message.revision2).should('exist');

      stageHistoryPage.revision.field.set('Revision 1');
      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '1'));
      stageHistoryPage.revision.empty.should('exist');

      stageHistoryPage.applyAsIs(1);

      stageHistoryPage.formButtons.cancel.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '1'));
      stageHistoryPage.revision.empty.should('exist');

      cy.log('откат к ревизии 1');
      stageHistoryPage.revision.field.check('Revision 1');
      stageHistoryPage.applyAsIs(1);

      stageHistoryPage.diffView.descriptionField.set(message.revision3);
      stageHistoryPage.diffView.deploy();

      cy.log('попадаем на страницу статуса, теперь надо проверить, применились ли изменения при откате');
      stageEditPage.visit();

      cy.get(duEditForm.treeItem.linkSelector).click();

      // FIXME: в классе локаций дописать проверки
      // duForm.form.check({ locations: STAGE_LOCATIONS.map(() => ({ podCount: podCount.revision1 })) });
      duEditForm.form.check({ disruptionBudget: String(disruptionBudget.revision1) });
      // @nodejsgirl TODO: во второй ревизии тоже надо проверять + надо добавить readonly проверки (даже если не сейчас, то хотя бы TODO надо добавить)

      cy.log('теперь проверим новую ревизию в истории');
      stageHistoryPage.visit();

      const revisionRow3 = stageHistoryPage.getRevisionRow(3);
      revisionRow3.currentRevision.should('exist');
      revisionRow3.message(message.revision3).should('exist');

      revisionRow3.row.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '3'));
      stageHistoryPage.revision.empty.should('not.exist');

      stageHistoryPage.revision.field.set('Revision 2');
      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '2'));
      stageHistoryPage.revision.text(message.revision2).should('exist');
      stageHistoryPage.revision.empty.should('not.exist');

      /******************************************
       * откат с изменениями *
       ******************************************/
      cy.log('# откат с изменениями');
      stageHistoryPage.applyWithChanges(2);

      cy.log('переходы между экранами');
      stageHistoryPage.formButtons.cancel.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '2'));

      stageHistoryPage.applyWithChanges(2);

      cy.log('форма с изменениями');
      const rollbackForm = stageHistoryPage.revisionHugeForm('2');
      const workloadForm = rollbackForm.workload(duId, boxId, workloadId);

      cy.log('включаем поставку логов');
      cy.get(workloadForm.treeItem.linkSelector).click();
      workloadForm.form.set({ logs: true });

      cy.log('проверяем сохранение урла');
      stageHistoryPage.formButtons.update.click();
      cy.location('pathname').should('contain', ROUTES.stageHistoryApplyWithChanges(stageName, '2'));

      cy.log('проверяем сохранение значения в форме');
      cy.get(stageHistoryPage.diffView.buttons.edit).click();
      cy.get(workloadForm.treeItem.linkSelector).click();

      workloadForm.form.check({ logs: true });

      cy.log('деплоим версию отката с изменениями');

      stageHistoryPage.formButtons.update.click();
      stageHistoryPage.diffView.deploy(message.revision4);

      cy.log('проверяем, что в текущей ревизии применились изменения');

      stageEditPage.visit();
      cy.get(duEditForm.treeItem.linkSelector).click();

      cy.log('данные из ревизии 2');

      // FIXME: в классе локаций дописать проверки
      // duForm.form.check({ locations: STAGE_LOCATIONS.map(() => ({ podCount: podCount.revision2 })) });
      duEditForm.form.check({ disruptionBudget: String(disruptionBudget.revision2) });

      cy.log('отдельно включенные логи');

      const workloadEditForm = stageEditPage.hugeForm.workload(duId, boxId, workloadId);

      cy.get(workloadEditForm.treeItem.linkSelector).click();
      workloadEditForm.form.check({ logs: true });

      cy.log('последняя ревизия теперь 4');

      stageHistoryPage.visit();
      const revisionRow4 = stageHistoryPage.getRevisionRow(4);

      revisionRow4.row.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(stageName, '4'));

      stageHistoryPage.revision.empty.should('not.exist');
      stageHistoryPage.revision.text(message.revision4).should('exist');

      cy.log('переход к сравнению revision 3 <-> revision 4');

      stageHistoryPage.buttons.compareRevisions.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryDiff(stageName, '3', '4'));

      cy.log('проверка полей в diff view');

      // удаленные поля
      stageHistoryPage.diffView.checkDeletedFieldValue('revision', '3');
      stageHistoryPage.diffView.checkDeletedFieldValue('description', message.revision3);
      stageHistoryPage.diffView.checkDeletedFieldValue('replica_count', String(podCount.revision1));
      stageHistoryPage.diffView.checkDeletedFieldValue('max_unavailable', String(disruptionBudget.revision1));

      // добавленные поля
      stageHistoryPage.diffView.checkInsertedFieldValue('revision', '4');
      stageHistoryPage.diffView.checkInsertedFieldValue('description', message.revision4);
      stageHistoryPage.diffView.checkInsertedFieldValue('replica_count', String(podCount.revision2));
      stageHistoryPage.diffView.checkInsertedFieldValue('max_unavailable', String(disruptionBudget.revision2));
      stageHistoryPage.diffView.checkInsertedFieldValue('transmit_logs', 'true');

      cy.log('проверка смены порядка'); // https://st.yandex-team.ru/DEPLOY-4700

      stageHistoryPage.buttons.rearrangeRevisions.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryDiff(stageName, '4', '3'));
      stageHistoryPage.buttons.rearrangeRevisions.click();
      cy.location('pathname').should('eq', ROUTES.stageHistoryDiff(stageName, '3', '4'));

      stageHistoryPage.diffView.checkDeletedFieldValue('revision', '3');
      stageHistoryPage.diffView.checkDeletedFieldValue('description', message.revision3);

      stageHistoryPage.diffView.checkInsertedFieldValue('revision', '4');
      stageHistoryPage.diffView.checkInsertedFieldValue('description', message.revision4);

      stageHistoryPage.diffComparedRevision.field.set('Revision 1');
      cy.location('pathname').should('eq', ROUTES.stageHistoryDiff(stageName, '1', '3'));

      stageHistoryPage.diffView.checkDeletedFieldValue('revision', '1');
      stageHistoryPage.diffView.checkInsertedFieldValue('revision', '3');
      // в идеале revision_info.description
      stageHistoryPage.diffView.checkInsertedFieldValue('description', message.revision3);

      cy.log('проверка настроек отображения diff');

      stageHistoryPage.diff.viewType.set('split');
      stageHistoryPage.diff.viewType.set('unified');
      stageHistoryPage.diff.allExpanded.set(true);

      // TODO: пустые строки нельзя проверить в текущем DiffView
      // page.diffView.checkFieldValue('meta', '');
      // page.diffView.checkFieldValue('acl', '');
      stageHistoryPage.diffView.checkFieldValue('id', stageName);

      cy.log('проверка тултипов');

      stageHistoryPage.diffRevision.field.set('Revision 1');
      stageHistoryPage.diffRevision.summary.open();
      stageHistoryPage.diffRevision.summary.title('Revision 1').should('exist');
      stageHistoryPage.diffRevision.summary.empty.should('exist');
      stageHistoryPage.diffRevision.summary.close();

      stageHistoryPage.diffComparedRevision.summary.open();
      stageHistoryPage.diffComparedRevision.summary.title('Revision 3').should('exist');
      stageHistoryPage.diffComparedRevision.summary.text(message.revision3).should('exist'); // DEPLOY-4699

      cy.log('переход из тултипа на страницу ревизии');

      stageHistoryPage.diffComparedRevision.summary.showYaml();

      stageHistoryPage.revision.field.check('Revision 3');
      stageHistoryPage.revision.text(message.revision3).should('exist');

      cy.log('возврат к истории');

      stageHistoryPage.buttons.backToHistory.click();
      revisionRow1.message(EMPTY_VALUE).should('exist');
   });
});
