import { getStageName, SERVICE, SERVICES } from '../../../../utils';

import { StageConfigPage } from 'page_objects/stage/StageConfigPage';
import { StageEditPage } from 'page_objects/stage/StageEditPage';
import { StageStatusPage } from 'page_objects/stage/StageStatusPage';
import { WorkloadCommandsFormValue } from 'page_objects/stage/internal/levels/workload/fields/WorkloadCommandsForm';

const resources = {
   nginx: {
      id: 'nginx',
      url: 'sbr:274084887',
      checksum: 'MD5:ea500ec8d56c2936bc98531052101e55',
      mount: '/mnt/nginx',
   },
   ui: {
      id: 'ui_bundle',
      url: 'sbr:2664830543',
      checksum: '', // disabled
      mount: '/mnt/ui_bundle',
   },
};

const commands: WorkloadCommandsFormValue[] = [
   {
      start: {
         command: 'bash /code/commands/start.sh',
      },
      init: [
         {
            order: 1,
            command: [
               'bash -c "',
               'mkdir -p /code',
               'tar xf /mnt/ui_bundle/*.tar.gz --directory=/code --strip-components=1',
               'tar xf /mnt/nginx/*.tar.gz --directory=/code/nginx',
               '"',
            ].join('\n'),
         },
      ],
   },
];

context('BoxStaticResources', () => {
   const stageName = getStageName('box_static_resources');

   const stageEditPage = new StageEditPage(stageName);
   const stageConfigPage = new StageConfigPage(stageName);
   const stageStatusPage = new StageStatusPage(stageName);

   const deployUnitForm = stageEditPage.hugeForm.deployUnit('deployUnit');
   const boxForm = stageEditPage.hugeForm.box('deployUnit', 'box');
   const workloadForm = stageEditPage.hugeForm.workload('deployUnit', 'box', 'workload');

   before(() => {
      cy.loginRobot();
      cy.emptyPage();

      // sure stage isn't exists
      cy.deleteStage(stageName);
   });

   after(() => {
      cy.deleteStage(stageName);
   });

   it('should edit static resources', () => {
      cy.createSimpleStage(stageName).then(() => {
         stageEditPage.visit();

         cy.get(workloadForm.treeItem.linkSelector).click();

         workloadForm.form.set({
            commands,
         });

         cy.get(deployUnitForm.treeItem.linkSelector).click();
         cy.get(deployUnitForm.formTabs.getTab('Disks, volumes and resources')).click();

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 1,
                  add: true,
               },
            ],
         });

         cy.log('Show disk static resources errors: empty static resource 1');
         stageEditPage.showValidationModal();
         stageEditPage.closeValidationModal();

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 1,
                  item: {
                     id: 'test',
                     url: 'sbr:1234567890',
                  },
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 1,
                  item: {
                     type: 'files',
                  },
               },
            ],
         });

         cy.log('Show disk static resources errors: "Static resource files should not be empty"');
         stageEditPage.showValidationModal();
         stageEditPage.closeValidationModal();

         // TODO: files tests

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 1,
                  remove: true,
               },
            ],
         });

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 1,
                  add: true,
               },
               {
                  order: 2,
                  add: true,
               },
            ],
         });

         deployUnitForm.form.check({
            staticResources: [
               {
                  order: 1,

                  item: {
                     verification: {
                        enabled: false, // default
                     },
                  },
               },
               {
                  order: 2,

                  item: {
                     verification: {
                        enabled: false, // default
                     },
                  },
               },
            ],
         });

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 2,

                  item: {
                     id: resources.nginx.id,
                     url: resources.nginx.url,

                     verification: {
                        enabled: true,
                        checksum: resources.nginx.checksum,
                     },
                  },
               },
               {
                  order: 3,

                  item: {
                     id: resources.ui.id,
                     url: resources.ui.url,
                  },
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 2,
                  item: {
                     accessPermissions: '660',
                  },
               },
               {
                  order: 2,
                  item: {
                     accessPermissions: 'unmodified',
                  },
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         cy.get(boxForm.treeItem.linkSelector).click();
         cy.get(boxForm.formTabs.getTab('Resources')).click();

         boxForm.form.set({
            layers: [
               {
                  order: 2,
                  remove: true,
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         boxForm.form.set({
            staticResources: [
               {
                  order: 1,
                  add: true,
               },
            ],
         });

         cy.log('Show box static resources errors: empty static resource 1');
         stageEditPage.showValidationModal();
         stageEditPage.closeValidationModal();

         boxForm.form.set({
            staticResources: [
               {
                  order: 2,
                  add: true,
                  item: { id: resources.nginx.id, mountPoint: resources.nginx.mount },
               },
               {
                  order: 1,
                  remove: true,
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         boxForm.form.set({
            staticResources: [
               {
                  order: 2,
                  add: true,
                  item: { id: resources.nginx.id, mountPoint: resources.nginx.mount },
               },
            ],
         });

         cy.log('Show box static resources errors: "Static resource must be unique"');
         stageEditPage.showValidationModal();
         stageEditPage.closeValidationModal();

         boxForm.form.set({
            staticResources: [
               {
                  order: 2,
                  item: { id: resources.ui.id, mountPoint: resources.ui.mount },
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         cy.get(deployUnitForm.treeItem.linkSelector).click();
         cy.get(deployUnitForm.formTabs.getTab('Disks, volumes and resources')).click();

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 2,
                  remove: true,
               },
               {
                  order: 3,
                  remove: true,
               },
            ],
         });

         cy.log(`Show disk static resources errors: "Static resource is used and cannot be removed."`);
         stageEditPage.showValidationModal();
         stageEditPage.closeValidationModal();

         deployUnitForm.form.set({
            staticResources: [
               {
                  order: 2,
                  restore: true,
               },
               {
                  order: 3,
                  restore: true,
               },
            ],
         });

         stageEditPage.showDiff();
         stageEditPage.closeDiff();

         stageEditPage.deploy();

         // check new form/config values
         [stageConfigPage, stageEditPage].map((page, pageIndex) => {
            page.visit();

            const deployUnit = page.hugeForm.deployUnit('deployUnit');
            const box = page.hugeForm.box('deployUnit', 'box');
            const workload = page.hugeForm.workload('deployUnit', 'box', 'workload');

            cy.get(deployUnit.treeItem.linkSelector).click();
            cy.get(deployUnitForm.formTabs.getTab('Disks, volumes and resources')).click();

            deployUnit.form.check({
               staticResources: [
                  {
                     order: 1,
                     item: {
                        id: resources.nginx.id,
                        type: 'url',
                        url: resources.nginx.url,
                        accessPermissions: 'unmodified',

                        verification: {
                           enabled: true,
                           checksum: resources.nginx.checksum,
                        },
                     },
                  },
                  {
                     order: 2,
                     item: {
                        id: resources.ui.id,
                        type: 'url',
                        url: resources.ui.url,
                        accessPermissions: 'unmodified',

                        verification: {
                           enabled: false,
                           checksum: resources.ui.checksum,
                        },
                     },
                  },
               ],
            });

            if (pageIndex === 1) {
               cy.log('Ошибок нет #RTCSUPPORT-16300');
               stageEditPage.showDiff();
               stageEditPage.closeDiff();
            }

            cy.get(box.treeItem.linkSelector).click();
            cy.get(boxForm.formTabs.getTab('Resources')).click();

            box.form.check({
               staticResources: [
                  {
                     order: 1,
                     item: { id: resources.nginx.id, mountPoint: resources.nginx.mount },
                  },
                  {
                     order: 2,
                     item: { id: resources.ui.id, mountPoint: resources.ui.mount },
                  },
               ],
            });

            cy.get(workload.treeItem.linkSelector).click();

            workload.form.check({
               commands,
            });
         });

         if (SERVICE === SERVICES.BACKEND) {
            // CONFIG_PRESET=ext-local
            stageStatusPage.stageReadyAfterSave('2');
         }
      });
   });
});
