import { dataTest, ROUTES, sel, TIMEOUTS } from '../../utils';

import { DiffView } from './internal/DiffView';
import { StageHugeFormButtons } from './internal/StageHugeFormButtons';
import { StageHugeForm } from './internal/StageHugeForm';
import { StagePatcherOptions } from './internal/StagePatcherOptions';

import { ValidationModal } from 'page_objects/components/ValidationModal';

import { StageTabPage } from './StageTabPage';
import { CheckboxField, YCRadioField, YCSelectField } from 'page_objects/components';
import { YCTooltip } from 'page_objects/components/internal/YCTooltip';

export class StageHistoryPage implements StageTabPage {
   public body = dataTest('StageHistory:List');

   public pathname = ROUTES.stageHistory(this.stageName);

   constructor(private stageName: string) {}

   public formButtons = new StageHugeFormButtons();

   public static selectors = {
      backToHistory: dataTest('StageHistory:BackHistoryButton'),
      backToRevision: dataTest('StageHistory:BackHistoryButton'),
      backToEdit: dataTest('StageHistory:BackHistoryButton'),
      compareRevisions: dataTest('StageHistory:CompareRevisions'),
      rearrangeRevisions: dataTest('StageHistory:RearrangeRevisions'),
      revisionId: dataTest('StageHistory:RevisionId'),
      messageBlock: dataTest('StageHistory:MessageBlock'),
      messageEmpty: dataTest('StageHistory:MessageEmpty'),
   };

   buttons = {
      get backToHistory() {
         return cy.get(StageHistoryPage.selectors.backToHistory);
      },
      get backToRevision() {
         return cy.get(StageHistoryPage.selectors.backToRevision);
      },
      get backToEdit() {
         return cy.get(StageHistoryPage.selectors.backToEdit);
      },
      get compareRevisions() {
         return cy.get(StageHistoryPage.selectors.compareRevisions);
      },
      get rearrangeRevisions() {
         return cy.get(StageHistoryPage.selectors.rearrangeRevisions);
      },
   };

   public revisionHugeForm = (revision: string) =>
      new StageHugeForm(ROUTES.stageHistoryApplyWithChanges(this.stageName, revision), false);

   public validationModal = new ValidationModal();

   public stagePatcherOptions = new StagePatcherOptions();

   public diffView = new DiffView();

   /**
    * строка ревизии в списке
    */
   public getRevisionRow(revisionId: number) {
      const revision = () => cy.get(StageHistoryPage.selectors.revisionId).contains(revisionId).parent('tr');

      return {
         /**
          * строка таблицы
          */
         get row() {
            return revision();
         },
         /**
          * пометка о текущей ревизии (верхняя в списке)
          */
         get currentRevision() {
            return revision().get(dataTest('StageHistory:CurrentRevision'));
         },
         /**
          * номер ревизии
          */
         get id() {
            return revision().get(StageHistoryPage.selectors.revisionId);
         },
         /**
          * время обновления
          */
         get time() {
            return revision().get(dataTest('StageHistory:RevisionTime'));
         },
         /**
          * автор обновления
          */
         get user() {
            return revision().get(dataTest('StageHistory:RevisionUser'));
         },

         /**
          * сообщение к ревизии
          */
         message: (value: string) => revision().get(dataTest('StageHistory:RevisionMessage')).contains(value),
      };
   }

   private getRevisionSummary(revisionSelector: string) {
      const body = () => cy.get(YCTooltip.selectors.body);
      const buttons = {
         get open() {
            return cy.get(sel(revisionSelector, dataTest('StageHistory:RevisionSummaryButton')));
         },
         get close() {
            return body().find(YCTooltip.selectors.closeButton);
         },
         get showYaml() {
            return body().find(dataTest('StageHistory:ShowYaml')).find('button');
         },
      };

      return {
         buttons,
         open: () => buttons.open.scrollIntoView().click(),
         close: () => buttons.close.click(),
         showYaml: () => buttons.showYaml.click(),
         title: (value: string) => body().find(YCTooltip.selectors.title).contains(value),
         text: (value: string) => body().find(StageHistoryPage.selectors.messageBlock).contains(value),
         get empty() {
            return body().find(StageHistoryPage.selectors.messageEmpty);
         },
      };
   }

   private getRevision(parentSelector: string) {
      return {
         field: new YCSelectField(sel(parentSelector, dataTest('StageHistory:SelectRevision'))),
         text: (value: string) => cy.get(StageHistoryPage.selectors.messageBlock).contains(value),
         get empty() {
            return cy.get(StageHistoryPage.selectors.messageEmpty);
         },
      };
   }

   private getDiffRevision(revisionSelector: string) {
      return Object.assign(this.getRevision(revisionSelector), {
         summary: this.getRevisionSummary(revisionSelector),
      });
   }

   /**
    * ревизия слева при сравнении (исходная)
    */
   public get diffRevision() {
      return this.getDiffRevision(dataTest('StageHistory:DiffRevision'));
   }

   /**
    * ревизия справа при сравнении (с изменениями)
    */
   public get diffComparedRevision() {
      return this.getDiffRevision(dataTest('StageHistory:DiffComparedRevision'));
   }

   /**
    * элементы страницы отдельной ревизии
    */
   public get revision() {
      return this.getRevision(dataTest('StageHistory:RevisionSpec'));
   }

   public diff = {
      viewType: new YCRadioField(dataTest('DiffView:Type')),
      allExpanded: new CheckboxField(dataTest('DiffView:FullContext')),
   };

   public visit() {
      cy.log(`Stage history page: visit ${this.pathname}`);
      cy.visit(this.pathname);
      cy.get(this.body, { timeout: TIMEOUTS.superSlow }).should('be.visible');
   }

   public showUpdates() {
      cy.log('Stage history page: show updates');
      this.formButtons.update.scrollIntoView().click();
      cy.get(this.stagePatcherOptions.wrapper).should('be.visible');
   }

   public showDiff() {
      cy.log('Stage history page: show diff');
      this.formButtons.update.scrollIntoView().click();
      cy.get(this.diffView.wrapper).should('be.visible');
   }

   public visitRevisionPage(revision: number) {
      cy.log(`Stage history revision page: visit ${this.pathname}`);

      this.visit();

      this.getRevisionRow(revision).row.click();

      cy.location('pathname').should('eq', ROUTES.stageHistoryRevision(this.stageName, String(revision)));
      cy.get(dataTest('StageHistory:RevisionSpec')).should('be.visible');
   }

   public applyAsIs(revision: number) {
      this.visitRevisionPage(revision);

      cy.get(dataTest('StageHistory:DiffApply')).should('be.visible').find('button').click();

      cy.get(dataTest('StageHistory:ApplyPopup'))
         .should('be.visible')
         .find(dataTest('StageHistory:ApplyAsIsLink'))
         .click();

      cy.location('pathname').should('eq', ROUTES.stageHistoryApplyAsIs(this.stageName, String(revision)));
   }

   public applyWithChanges(revision: number) {
      this.visitRevisionPage(revision);

      cy.get(dataTest('StageHistory:DiffApply')).should('be.visible').find('button').click();

      cy.get(dataTest('StageHistory:ApplyPopup'))
         .should('be.visible')
         .find(dataTest('StageHistory:ApplyWithChangesLink'))
         .click();

      cy.location('pathname').should('eq', ROUTES.stageHistoryApplyWithChanges(this.stageName, String(revision)));
   }
}
