import { dataE2E, dataQa, dataTest, sel } from '../../../../../../utils';

import { EnabledSwitcherField, EnumField, TextInputField, TextInputFieldValue } from '../../../../../components';

type Verification = Partial<{
   enabled: boolean;
   checksum: TextInputFieldValue;
}>;

export type DiskStaticResourceItem = Partial<{
   id: TextInputFieldValue;
   type: string;
   url: TextInputFieldValue;
   accessPermissions: string;
   verification: Verification;
}>;

export type Action = Partial<{
   add: boolean;
   remove: boolean;
   restore: boolean;
   item: DiskStaticResourceItem;
}>;

export interface DiskStaticResourceAction extends Action {
   order: number;
}

export class DiskStaticResourcesForm {
   constructor(private selector: string) {}

   private buttons = {
      add: sel(this.selector, dataQa('StaticResources:AddStaticResource')),
   };

   private cardForm = (order: number) => {
      const card = sel(this.selector, dataTest(`StaticResource:${order}`));
      const n = order - 1;

      return {
         fields: {
            idField: new TextInputField(sel(card, dataQa(`disks.[0].staticResources[${n}].id:FieldLayout`))),
            typeField: new EnumField(sel(card, dataQa(`disks.[0].staticResources[${n}].type:FieldLayout`))),
            urlField: new TextInputField(sel(card, dataQa(`disks.[0].staticResources[${n}].url:FieldLayout`))),

            accessPermissionsField: new EnumField(
               sel(card, dataQa(`disks.[0].staticResources[${n}].accessPermissions:FieldLayout`)),
            ),

            validation: {
               enabledField: new EnabledSwitcherField(
                  sel(card, dataQa(`disks.[0].staticResources[${n}].verification.enabled:FieldLayout`)),
               ),

               checksumField: new TextInputField(
                  sel(card, dataQa(`disks.[0].staticResources[${n}].verification.checksum:FieldLayout`)),
               ),
            },
         },
         buttons: {
            remove: sel(card, dataE2E('StaticResource:Remove')),
            restore: sel(card, dataE2E('StaticResource:Restore')),
         },
      };
   };

   private add() {
      cy.log(`Add new static resource`);
      cy.get(this.buttons.add).scrollIntoView().should('be.visible').click();
   }

   public set(value: DiskStaticResourceAction[]) {
      this.apply(value, 'set');
   }

   public check(value: DiskStaticResourceAction[], ...params) {
      this.apply(value, 'check', ...params);
   }

   protected apply(value: DiskStaticResourceAction[], methodName: string, ...params) {
      value.forEach(action => {
         cy.log(`Disk static resource ${action.order} settings`);

         const { fields, buttons } = this.cardForm(action.order);

         if (action.add) {
            this.add();
         }

         if (action.item) {
            if (action.item.id) {
               fields.idField[methodName](action.item.id, ...params);
            }

            if (action.item.type) {
               fields.typeField[methodName](action.item.type, ...params);
            }

            if (action.item.url) {
               fields.urlField[methodName](action.item.url, ...params);
            }

            if (action.item.accessPermissions) {
               fields.accessPermissionsField[methodName](action.item.accessPermissions, ...params);
            }

            if (action.item.verification) {
               if (action.item.verification.enabled) {
                  fields.validation.enabledField[methodName](action.item.verification.enabled, ...params);
               }

               if (action.item.verification.checksum) {
                  fields.validation.checksumField[methodName](action.item.verification.checksum, ...params);
               }
            }
         }

         if (action.remove) {
            cy.log(`Remove static resource`);
            cy.get(buttons.remove).should('be.visible').click();
         }

         if (action.restore) {
            cy.log(`Restore static resource`);
            cy.get(buttons.restore).should('be.visible').click();
         }
      });
   }
}
