import { omitBy } from 'lodash';

export const DEFAULT_PROJECT = 'deploy-e2e';
export const DEFAULT_DEPLOY_UNIT = 'deployUnit';
export const DEFAULT_BOX = 'box';
export const DEFAULT_WORKLOAD = 'workload';

export const DEFAULT_ABC_SERVICE = 'Я.Деплой';
export const DEFAULT_ACCOUNT_ID = 'abc:service:3494';

export const DEFAULT_NETWORK = '_YA_DEPLOY_NETS_';

export const UI_PROJECT_NAME = 'deploy-ui';
export const UI_STAGE_NAMES = ['man-pre-deploy-ui', 'pre-deploy-ui', 'pre2-deploy-ui', 'test-deploy-ui'];
export const UI_STAGE_NAMES_NO_MATCH = UI_STAGE_NAMES.map(name =>
   UI_STAGE_NAMES.find(v => v !== name && !v.includes(name)),
);

export const BYTES = {
   B: 1024 ** 0,
   KB: 1024 ** 1,
   MB: 1024 ** 2,
   GB: 1024 ** 3,
   TB: 1024 ** 4,
};

export const TIMES_IN_S = {
   Second: 1,
   Minute: 60,
   Hour: 60 * 60,
   Day: 60 * 60 * 24,
   Week: 60 * 60 * 24 * 7,
};

export const TIMES_IN_MS = {
   Millisecond: 1,
   Second: 1000 * TIMES_IN_S.Second,
   Minute: 1000 * TIMES_IN_S.Minute,
   Hour: 1000 * TIMES_IN_S.Hour,
   Day: 1000 * TIMES_IN_S.Day,
   Week: 1000 * TIMES_IN_S.Week,
};

export const TIMEOUTS = {
   // presets

   /* 1 c - элемент уже должен быть на странице */
   immediately: TIMES_IN_MS.Second,

   /* 5 c - элемент либо уже есть, либо вот-вот будет (легкий API-запрос) */
   fast: 5 * TIMES_IN_MS.Second,

   /* 10 c - элемент скоро будет (тяжелый API-запрос) */
   slow: 10 * TIMES_IN_MS.Second,

   /* некоторые API-запросы "залипают", не укладываясь в 10 с (например, deleteProject) */
   superSlow: 30 * TIMES_IN_MS.Second,

   api: {
      // временный таймаут с большим запасом, т.к. падают тесты на саджестах в сандбоксе
      // AssertionError: Timed out retrying after 30000ms: Expected to find element: `[data-test="form-field--abc-service"] .suggest .scrollable-list__item`, but never found it.
      //    at Context.eval (https://localhost.yandex-team.ru:9000/__cypress/tests?p=integration/userStories/project/ProjectCreate.ts:18027:16)
      // AssertionError: Timed out retrying after 30000ms: Expected to find element: `.yc-suggest__popup .yc-suggest__list .yc-suggest__item div`, but never found it.
      //    at YCSuggestField.eval (https://localhost.yandex-team.ru:9000/__cypress/tests?p=integration/userStories/stage/StageCreate.ts:18108:20)
      suggest: 3 * TIMES_IN_MS.Minute,
      // AssertionError: Timed out retrying after 10000ms: Expected to find element: `h1.project__title`, but never found it.
      //    at ProjectPage../page_objects/project/ProjectPage.ts.ProjectPage.waitForProjectACL (https://localhost.yandex-team.ru:9000/__cypress/tests?p=integration/userStories/project/ProjectDelete.ts:18178:12)
      //    at Context.eval (https://localhost.yandex-team.ru:9000/__cypress/tests?p=integration/userStories/project/ProjectDelete.ts:177:21)
      pageLoading: 2 * TIMES_IN_MS.Minute,
   },

   ypRetries: 5 * TIMES_IN_MS.Minute,

   // forced. Значения, больше 30 секунд должны определятся явно для каждого случая.
   /**
    * Андрей Староверов:
    *
    * Поставьте 20 сек, если можете. Главное не создавать новый стейдж сразу после удаления.
    *
    * Это недостаток наследования ACL через stagectl, которое уйдет при переходе на наследование в YP.
    *
    * Если ты удалишь сразу после создания, то тоже ошибка будет,
    * потому что контроллеры не успеют твой стейдж подхватить, но это ретраями лечится хотя бы.
    */
   beforeRecreateStage: 60 * TIMES_IN_MS.Second,

   /**
    * Андрей Староверов:
    *
    * Можешь удалять проект не сразу после создания, а подождать секунд 20-30? 30с точно должно хватить.
    * Там суть в том, что Auth проиндексировал создание, а когда захотел роль накинуть, то проект уже удалили
    */
   beforeDeleteNewProject: TIMES_IN_MS.Minute,

   // https://st.yandex-team.ru/DEPLOY-4337
   beforeCreateNewProject: TIMES_IN_MS.Minute, // могут приехать старые ACL #DEPLOY-5035

   // https://st.yandex-team.ru/DEPLOY-4281
   // Исправится после DEPLOY-4683
   // beforeUpdateStage: 10 * TIMES_IN_MS.Second,

   stageReadyAfterSave: 15 * TIMES_IN_MS.Minute,
   afterPassport: 2 * TIMES_IN_MS.Second,
   waitStageAcl: 30 * TIMES_IN_MS.Second,
   waitProjectAcl: 5 * TIMES_IN_MS.Minute,

   // AssertionError: Timed out retrying after 300000ms: expected '<input#checked.yc-checkbox__control>' not to be 'disabled'
   //    at Context.eval (https://localhost.yandex-team.ru:9000/__cypress/tests?p=integration/userStories/stage/StageDelete.ts:200:18)
   waitStageDeletionCheckbox: 5 * TIMES_IN_MS.Minute,

   // форма не должна менятся на период внешних обновлений объектов yp
   // https://st.yandex-team.ru/SPI-36038
   waitFormStableRender: TIMES_IN_MS.Minute,
};

export enum SERVICES {
   BACKEND = 'backend',
   FRONTEND = 'frontend',
}

export const SERVICE: SERVICES = Cypress.env('service') ?? SERVICES.FRONTEND;

export const DISABLED_LOCATIONS = Cypress.env('disabledLocations') ?? ['myt'];

export const STAGE_LOCATIONS: string[] = (Cypress.env('locations') ?? ['sas', 'man', 'vla', 'iva', 'myt']).filter(
   v => !DISABLED_LOCATIONS.includes(v),
);

export const STAGE_SPEC_MCRS_CLUSTERS = STAGE_LOCATIONS.map(v => ({
   'cluster': v,
   'spec': {
      'replica_count': 1,
   },
}));

const ENTITY_PREFIX = Cypress.env('entityPrefix') ?? Cypress.env('ROBOT_LOGIN') ?? 'robot-infracloudui';
const COMMIT_PREFIX = Cypress.env('commitPrefix') ?? '';

export function getStageName(stageName: string) {
   return `e2e_${ENTITY_PREFIX}_${COMMIT_PREFIX}_${stageName}`;
}

export function getProjectName(projectName: string) {
   return `e2e_${ENTITY_PREFIX}_${COMMIT_PREFIX}_${projectName}`;
}

const removeEndSlashes = (s: string) => s.slice(0, s.search(/\/*$/));
const storybookHost = removeEndSlashes(Cypress.env('storybookHost'));

export function getStoryRawUrl(story: string) {
   return `${storybookHost}/iframe.html?id=${story}`;
}

export function getStoryUrl(id: string, prefix = 'cypress') {
   return getStoryRawUrl(`${prefix}--${Cypress._.kebabCase(id)}`);
}

export const ROUTES = {
   stage(stageName: string) {
      return `/stages/${stageName}`;
   },
   stageBalancers(stageName: string) {
      return `${this.stage(stageName)}/balancers`;
   },
   stageConfig(stageName: string) {
      return `${this.stage(stageName)}/config`;
   },
   stageDeployTickets(stageName: string) {
      return `${this.stage(stageName)}/deploy-tickets`;
   },
   stageEdit(stageName: string) {
      return `${this.stage(stageName)}/edit`;
   },
   stageHistory(stageName: string) {
      return `${this.stage(stageName)}/history`;
   },
   stageHistoryApplyAsIs(stageName: string, revision: string) {
      return `${this.stage(stageName)}/history/${revision}/applyAsIs`;
   },
   stageHistoryApplyWithChanges(stageName: string, revision: string) {
      return `${this.stage(stageName)}/history/${revision}/apply`;
   },
   stageHistoryDiff(stageName: string, revision1: string, revision2: string) {
      return `${this.stage(stageName)}/history/${revision1}/diff/${revision2}`;
   },
   stageHistoryRevision(stageName: string, revision: string) {
      return `${this.stage(stageName)}/history/${revision}`;
   },
   stageLogs(stageName: string) {
      return `${this.stage(stageName)}/logs`;
   },
   stageMonitoring(stageName: string) {
      return `${this.stage(stageName)}/monitoring`;
   },
   stageStatus(stageName: string) {
      return `${this.stage(stageName)}`;
   },
   stageUpdate(stageName: string) {
      return `${this.stage(stageName)}/update`;
   },

   project(projectName: string) {
      return `/projects/${projectName}`;
   },
   projects() {
      return `/?my=yes`;
   },

   newStage(projectId?: string) {
      return `/new-stage${this.buildQuery({ projectId })}`;
   },

   stageFormInternal(root: string, deployUnitId?: string, boxId?: string, workloadId?: string) {
      let result = root;

      if (deployUnitId) {
         result += `/du-${deployUnitId}`;

         if (boxId) {
            result += `/box-${boxId}`;

            if (workloadId) {
               result += `/wl-${workloadId}`;
            }
         }
      }

      return result;
   },

   buildQuery<P extends Record<string, any>>(params: P | undefined) {
      if (!params) {
         return '';
      }

      const cleanedParams = omitBy(params, v => v === undefined);
      const queryString = Object.keys(cleanedParams)
         .map(k => {
            const v = cleanedParams[k];

            return `${encodeURIComponent(k)}=${encodeURIComponent(v)}`;
         })
         .join('&');

      return queryString ? `?${queryString}` : '';
   },

   home() {
      return this.projects();
   },
};

export const SECRET = {
   name: 'aaa_testcafe_deploy',
   keys: ['PORT', 'TVM_SOURCE_1', 'TVM_SOURCE_2'],
};

export const EMPTY_VALUE = '\u2014';
