const { exec } = require('child_process');
const { v4 } = require('uuid');
const dateFns = require('date-fns');
const fs = require('fs');

const { fixes } = require('./commit_message_fixes.js');

const uniqueSeparator = v4();
const gitLogFormat = `${uniqueSeparator}%n%H%n%P%n%ae%n%aD%n%D%n%B`;

// отслеживаемые очереди
const tickets = ['DEPLOY', 'RTCSUPPORT'];

const ticketRegex = new RegExp(`(${tickets.join('|')})-\\d+`, 'mg');

const tagRegex = /tag: ui_\d+\.\d+\.\d+/;

// синхронный вариант вызывает ошибку из-за большого количества данных
exec(`git log --pretty="format:${gitLogFormat}"`, { maxBuffer: 1024 * 2000 }, (error, stdout) => {
   if (error) {
      console.error(error);
   }
   const log = stdout;
   const commits = [];
   const values = log.split(`${uniqueSeparator}\n`).filter(Boolean);

   // исправление сообщений

   for (const value of values) {
      const [hash, parents, email, date, refs, ...body] = value.split('\n');
      let message = body.join('\n');
      if (hash in fixes) {
         message = fixes[hash](message);
      }
      commits.push({
         hash,
         parents: parents.split(' '),
         email,
         date,
         refs,
         message,
      });
   }

   let lastVersion;

   const changelog = {};

   // коммиты предки для каждого коммита головы
   const heads = {};

   // данные изменений по коммитам
   const commitChanges = {};

   // коммиты, уже включенные в историю ревизий
   const historyCommits = new Set();

   // слияние деревьев
   function getFullTree(hashes) {
      const resultTickets = {};
      for (const hash of hashes) {
         // собираем тикеты по всем коммитам
         const commitTickets = commitChanges[hash];
         for (const ticketId of Object.keys(commitTickets)) {
            if (!resultTickets[ticketId]) {
               resultTickets[ticketId] = {};
            }
            const ticket = resultTickets[ticketId];
            for (const author of Object.keys(commitTickets[ticketId])) {
               if (!ticket[author]) {
                  ticket[author] = {};
               }
               const dates = ticket[author];
               for (const date of Object.keys(commitTickets[ticketId][author])) {
                  if (!dates[date]) {
                     dates[date] = [];
                  }
                  dates[date].push(...commitTickets[ticketId][author][date]);
               }
            }
         }
      }
      return resultTickets;
   }

   // появились новые тикеты, не записанные в историю
   function existNewTickets() {
      return Object.keys(commitChanges).some(hash => !historyCommits.has(hash));
   }

   function addVersionChanges(version, hash, date) {
      // все предки версии
      const allParents = heads[hash];
      // все новые с момента предыдущей версии
      const newParents = new Set([...allParents].filter(_hash => !historyCommits.has(_hash)));
      // те из них, для которых есть записи
      const currentChanges = [...newParents].filter(_hash => _hash in commitChanges);

      if (currentChanges.length > 0) {
         const versionTickets = getFullTree(currentChanges);
         for (const newHash of currentChanges) {
            // помечаем коммит как записанный в историю
            historyCommits.add(newHash);
         }

         changelog[version] = { ...versionTickets, __date: date };
      }
   }

   for (let i = commits.length - 1; i >= 0; i -= 1) {
      const { message, refs, date: rawDate, email, hash, parents } = commits[i];

      heads[hash] = new Set();

      for (const parent of parents) {
         if (parent in heads) {
            // добавляем коммит и всех предков
            const scope = new Set(heads[parent]);
            scope.add(parent);
            heads[hash] = new Set([...scope, ...(heads[hash] || [])]);
         } else {
            // на случай сломанной истории
            heads[parent] = new Set();
         }
      }

      const messageTickets = message.match(ticketRegex) || [];

      const currentTickets = {};

      if (/.+@yandex-team\.ru/.test(email)) {
         for (const ticketId of messageTickets) {
            if (!currentTickets[ticketId]) {
               currentTickets[ticketId] = {};
            }
            const ticket = currentTickets[ticketId];
            const author = email.split('@')[0];
            if (!ticket[author]) {
               ticket[author] = {};
            }

            const [date, time] = dateFns.format(new Date(rawDate), 'dd.MM.yyyy HH:mm').split(' ');

            const dates = ticket[author];

            if (!dates[date]) {
               dates[date] = [];
            }

            dates[date] = [...new Set([...dates[date], time])];
         }
      }

      if (Object.keys(currentTickets).length > 0) {
         commitChanges[hash] = currentTickets;
      }

      const tags = refs.match(tagRegex);
      if (tags) {
         const version = tags[0].slice(8);
         lastVersion = version;
         // нет такой версии и есть что нового записать
         if (!changelog[version] && existNewTickets()) {
            addVersionChanges(version, hash, rawDate);
         }
      }
   }

   const currentVersion = JSON.parse(fs.readFileSync(`${__dirname}/../package.json`).toString()).version;
   if (currentVersion !== lastVersion && existNewTickets()) {
      addVersionChanges(currentVersion, commits[0].hash, commits[0].date);
   }

   console.log(JSON.stringify(changelog));
});
