const prettier = require('prettier');
const ejs = require('ejs');
const formatDate = require('date-fns/lightFormat');
const numeral = require('numeral');
const { promisify } = require('util');

const render = promisify(ejs.renderFile.bind(ejs));

function removeCallstack(errorMessage) {
   return errorMessage.replace(/^\s* at .+?$\n?/gm, '').trim();
}

function renderRow(width = 80, char = '=') {
   return new Array(width).fill(char).join('');
}

function renderHeader(text, width = 80) {
   const row = renderRow(width);
   const shift = text.length >= width ? 0 : Math.floor((width - text.length) / 2);

   const spaces = new Array(shift).fill(' ').join('');

   return `${row}\n${spaces}${text.toUpperCase()}\n${row}`;
}

function highlightDiff(text) {
   return text
      .split('\n')
      .map(row => {
         if (row.startsWith('@')) {
            return `<span class="d">${row}</span>`;
         }

         if (row.startsWith('+')) {
            return `<span class="r">${row}</span>`;
         }

         if (row.startsWith('-')) {
            return `<span class="e">${row}</span>`;
         }

         return row;
      })
      .join('\n');
}

// https://core.telegram.org/bots/api#markdownv2-style
// const escapedChars = ['_', '*', '\\[', '\\]', '(', ')', '~', '`', '>', '#', '+', '-', '=', '|', '{', '}', '.', '!'];
// const escapedCharsRe = new RegExp(`[${escapedChars.join('')}]`, 'g');

function escapeMd2(text) {
   return text.replace(/_/g, '\\_').replace(/-/g, '\\-');
}

class ReportRenderer {
   constructor(results) {
      const suiteTestResult = results.testResults[0];
      const duration = suiteTestResult.perfStats.end - suiteTestResult.perfStats.start;
      const allTests = suiteTestResult.testResults;

      const isStageTest = tr => tr.fullName.startsWith('AllStages tests Check all stages from the dump');
      const isSkipTest = tr => tr.fullName.startsWith('AllStages tests Check skipList actuality');

      const isSuccessful = tr => tr.status === 'passed';
      const isFailed = tr => tr.status === 'failed';
      const isSkipped = tr => tr.status === 'pending';

      const getTestResult = typePredicate => {
         const tests = allTests.filter(typePredicate);

         return {
            failed: tests.filter(isFailed),
            passed: tests.filter(isSuccessful),
            skipped: tests.filter(isSkipped),
            total: tests.length,
         };
      };

      const reportDate = formatDate(suiteTestResult.perfStats.end, 'yyyy-MM-dd');

      this.data = {
         duration: numeral(duration / 1000).format('00:00'),
         endedAt: formatDate(suiteTestResult.perfStats.end, 'yyyy-MM-dd HH:mm:ss'),
         escapeMd2,
         removeCallstack,
         highlightDiff,
         renderHeader,
         renderRow,
         reportDate,
         reportFileName: `all-stages-report.${reportDate}.html`,
         skipList: getTestResult(isSkipTest),
         stages: getTestResult(isStageTest),
         startedAt: formatDate(suiteTestResult.perfStats.start, 'yyyy-MM-dd HH:mm:ss'),
      };
   }

   async jsonSummary() {
      return {
         ok:
            this.data.stages.total !== 0 &&
            this.data.stages.failed.length === 0 &&
            this.data.skipList.failed.length === 0,
         reportDate: this.data.reportDate,
         reportFileName: this.data.reportFileName,
      };
   }

   async renderText() {
      return render(`${__dirname}/text/root.ejs`, this.data);
   }

   async renderHtml() {
      const html = await render(`${__dirname}/html/root.ejs`, this.data);

      return prettier.format(html, { parser: 'html' });
   }

   async renderMarkdown() {
      const md = await render(`${__dirname}/tg_md/root.ejs`, this.data);

      return md.replace(/\n{3,}/g, '\n\n');
   }
}

module.exports = { ReportRenderer };
