import { lstatSync, readdirSync, readFileSync, writeFileSync } from 'fs';
import * as prettier from 'prettier';
import * as ts from 'typescript';

const strType = (e: any) => `'${e}'`;

const keys: Record<string, Set<string>> = {};
const checkKeys = (name: string) => {
   if (!keys[name]) {
      keys[name] = new Set();
   }
};
const attrMask = 'qa|data-(e2e|test|qa)';
const attrMaskRegex = new RegExp(attrMask);

function findKeys(node: ts.Node) {
   if (!ts.isJsxAttribute(node)) {
      ts.forEachChild(node, findKeys);
      return;
   }

   const name = node.name.text;

   if (name === 'controlAttrs') {
      findInObject(node);
      return;
   }
   if (!attrMaskRegex.test(name)) {
      return;
   }

   const { initializer } = node;
   if (!initializer) {
      return;
   }

   // строковый атрибут
   if (initializer.kind === ts.SyntaxKind.StringLiteral) {
      checkKeys(name);
      keys[name].add(strType(initializer.text));
      return;
   }

   // выражение в фигурных скобках
   const { expression } = initializer;
   if (!expression) {
      return;
   }

   // строковый литерал
   if (ts.isStringLiteral(expression)) {
      checkKeys(name);
      keys[name].add(strType(expression.text));
      return;
   }

   // шаблонный литерал без содержимого
   if (ts.isNoSubstitutionTemplateLiteral(expression)) {
      checkKeys(name);
      keys[name].add(strType(expression.text));
      return;
   }

   // шаблонный литерал с содержимым
   if (ts.isTemplateExpression(expression)) {
      // eslint-disable-next-line no-template-curly-in-string
      const typeView = expression.getText().replace(/\$\{.+}/g, '<string>');
      // eslint-disable-next-line no-template-curly-in-string
      if (typeView === '`<string>`') {
         return;
      }
      checkKeys(name);
      keys[name].add(typeView);
   }
}

const attrMaskPropertyRegex = new RegExp(`'(${attrMask})': '(.+)'`);

function findInObject(node: ts.JsxAttribute) {
   const { initializer } = node;
   if (!initializer) {
      return;
   }

   // строковый атрибут
   if (initializer.kind === ts.SyntaxKind.StringLiteral) {
      return;
   }

   // выражение в фигурных скобках
   const { expression } = initializer;
   if (!expression) {
      return;
   }

   if (ts.isObjectLiteralExpression(expression)) {
      const text = expression.getText();
      const match = text.match(attrMaskPropertyRegex);

      if (match) {
         const name = match[1];
         checkKeys(name);
         keys[name].add(strType(match[3]));
      }
   }
}

const mask = /\.([tj])sx?$/;
const disableMask = /^node_modules$/;

const check = (dirPath: string) => {
   for (const name of readdirSync(dirPath)) {
      // console.log(name);
      const path = `${dirPath}/${name}`;
      const stats = lstatSync(path);
      if (stats.isFile()) {
         if (mask.test(name)) {
            find(path);
         }
      } else if (stats.isDirectory() && !disableMask.test(name)) {
         check(path);
      }
   }
};

function find(fileName) {
   // Parse a file
   const sourceFile = ts.createSourceFile(
      fileName,
      readFileSync(fileName).toString(),
      ts.ScriptTarget.ES5,
      /* setParentNodes */ true,
   );

   findKeys(sourceFile);
}

check(`${__dirname}/../${process.argv[2]}`);

const typesText = `export type TestKeys = {
${Object.keys(keys)
   .sort()
   .map(
      keyName =>
         `\t'${keyName}':${Array.from(keys[keyName])
            .sort()
            .map(e => `\n\t\t| ${e}`)
            .join('')}\n\t\t| (string & {});`,
   )
   .join('\n')}
};
`;

prettier.resolveConfig(__dirname).then(prettierConfig => {
   process.argv.slice(3).forEach(path => {
      writeFileSync(`${__dirname}/../${path}`, prettier.format(typesText, { parser: 'typescript', ...prettierConfig }));
   });
});

export {};
