import { isEmpty, isEqual, sortHandler, unique } from '@yandex-infracloud-ui/libs';
import * as fs from 'fs';

import {
   DeployUnitConverter,
   DeployUnitType,
   DiskLayerType,
   dockerNameComponentRegEx,
   dockerTagRegEx,
   StageConverter,
   StaticResourceFiles,
   StaticResourceFileType,
   StaticResourceType,
} from '../models/ui';
import { SecretUsage } from '../models/ui/secrets';
import { isSecretAliasGenerated } from '../modules/secrets/helpers';
import { EResourceAccessPermissions, TPodSpec_TIP6AddressRequest, TStage, TTvmConfig_EMode } from '../proto-typings';

describe.skip('DumpResearch', () => {
   // eslint-disable-next-line @typescript-eslint/no-var-requires,global-require
   const response = require('../../scripts/all_stages.json');

   const stagesMap = new Map<string, TStage>(
      response.results.map(({ values }: any) => {
         const [meta, labels, spec] = values;
         const stage = { meta, labels, spec };

         return [stage.meta.id, stage];
      }),
   );

   describe('Secrets', () => {
      it('Secrets by stage', () => {
         const usageCount = new Map<string, number>();
         stagesMap.forEach(stage => {
            const model = StageConverter.fromApi(stage);
            usageCount.set(
               model.id,
               model.deployUnits.reduce((acc, du) => acc + DeployUnitConverter.getSecretUsages(du, null).length, 0),
            );
         });

         const records = Array.from(usageCount.entries()).sort((a, b) => sortHandler(b[1], a[1]));
         console.log(records);
      });

      it('Get stages with duplicated secrets', () => {
         interface ResultRecord {
            stage: string;
            du: {
               [duId: string]: string[];
            };
         }

         const result: ResultRecord[] = [];

         stagesMap.forEach(stage => {
            const model = StageConverter.fromApi(stage);
            const record: ResultRecord = { stage: model.id, du: {} };

            for (const du of model.deployUnits) {
               const duProblems: string[] = [];
               const podTemplateSpec = DeployUnitConverter.getPodTemplateSpec(stage.spec!.deploy_units[du.id]);

               const podSecretMap = podTemplateSpec?.spec?.secrets ?? {};
               const aliases = new Set(Object.keys(podSecretMap));
               const knownVersions = new Set<string>();

               for (const alias of aliases) {
                  const secretRecord = podSecretMap[alias];
                  const version = `${secretRecord.secret_id}:${secretRecord.secret_version}`;
                  if (knownVersions.has(version)) {
                     duProblems.push(`Duplicated secret version "${version}" in "${alias}" record`);
                  } else {
                     knownVersions.add(version);
                  }

                  if (isSecretAliasGenerated(alias) && version !== alias) {
                     // duProblems.push(`Expected "${version}" but actual "${alias}"\n`);
                  }
               }

               if (duProblems.length > 0) {
                  record.du[du.id] = duProblems;
               }
            }

            if (!isEmpty(record.du)) {
               result.push(record);
            }
         });

         let log = '';
         for (const record of result) {
            log += `Stage "${record.stage}"\n`;
            for (const duId of Object.keys(record.du)) {
               log += `\tDU "${duId}\n`;
               log += `\t\t${record.du[duId].join('\n\t\t')}\n`;
            }
            log += '\n';
         }

         log += `Total stages: ${result.length}`;
         log += '\n';

         console.log(log);

         fs.writeFileSync('report.txt', log);
      });

      it('Get stages with semi-filled secrets', () => {
         interface ResultRecord {
            stage: string;
            du: {
               [duId: string]: SecretUsage[];
            };
         }

         const result: ResultRecord[] = [];
         stagesMap.forEach(stage => {
            const model = StageConverter.fromApi(stage);
            const record: ResultRecord = { stage: model.id, du: {} };

            for (const du of model.deployUnits) {
               const usages = DeployUnitConverter.getSecretUsages(du, null).filter(u => !u.key);

               if (usages.length > 0) {
                  record.du[du.id] = usages;
               }
            }

            if (!isEmpty(record.du)) {
               result.push(record);
            }
         });

         console.log(JSON.stringify(result, null, 2));
      });
   });

   it('Static resources by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const staticResources = model.deployUnits
            .flatMap(v => v.disks)
            .flatMap(box => box.staticResources)
            .filter(v => v && v.type === StaticResourceType.Files && v.files !== undefined)
            .map(v => v as StaticResourceFiles);

         // TODO: #DEPLOY-5283
         // const hasRawAndSecretFiles = staticResources.find(
         //    r =>
         //       r.files?.some(f => f.type === StaticResourceFileType.Raw && f.raw) &&
         //       r.files?.some(f => !(f.type === StaticResourceFileType.Raw && f.raw)),
         //       // r.files?.some(f => f.type === StaticResourceFileType.Secret && f.secret),
         // );
         // if (hasRawAndSecretFiles) {
         //    console.log(hasRawAndSecretFiles)
         //    result.push(model.id);
         // }

         // TODO: #DEPLOY-5284
         const { soxService } = model;
         const hasBadPermissions = staticResources.some(
            r =>
               r.accessPermissions === EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED &&
               r.files?.some(f => f.type === StaticResourceFileType.Secret && f.secret),
         );
         if (soxService && hasBadPermissions) {
            result.push(model.id);
         }

         // 2 types
         // const exist = staticResources.find(r => r.files?.some(f => f.raw) && r.files?.some(f => f.secret));

         // 1 type
         // const exist = staticResources.find(r => r.files?.some(f => f.type === StaticResourceFileType.Raw));
         // const exist = staticResources.find(r => r.files?.some(f => f.type === StaticResourceFileType.Secret));

         // meta
         // const exist = staticResources.find(r => r.meta);
         // const exist = staticResources.find(r => r.files && r.checksum !== 'EMPTY:');

         // const exist = staticResources.find(r => !isEmpty(r.files) && r.files.length > 2);
         // const exist = staticResources.find(r => !isEmpty(r.files) && r.files.length > 10 && r.files.length < 80);
         // const exist = staticResources.find(r => !isEmpty(r.files) && r.files.some(f => f.raw && f.raw?.length > 1000));

         // // duplicated files
         // const exist = staticResources.find(r => {
         //    if (!isEmpty(r.files)) {
         //       const names = r.files.map(v => v.name?.toUpperCase());

         //       if (new Set(names).size !== names.length) {
         //          console.log(names);
         //          return true;
         //       }
         //    }

         //    return false;
         // });

         // if (exist) {
         //    // if (exist && staticResources?.length > 1) {
         //    // console.log(exist?.files?.length)
         //    // console.log(exist?.files)
         //    // console.log(exist?.files?.find(f => f.raw && f.raw?.length > 600)?.raw?.length)
         //    // console.log(exist);
         //    result.push(model.id);
         // }

         // if (staticResources?.length === 0) {
         //    result.push(model.id);
         // }
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Layers by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const layers = model.deployUnits.flatMap(v => v.disks).flatMap(box => box.layers) ?? [];

         const unknown = layers.filter(v => v.type === DiskLayerType.Unknown);

         if (!isEmpty(unknown)) {
            result.push(model.id);
         }
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Custom ip6_address_requests by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const deployUnits = stage.spec?.deploy_units ?? {};

         Object.keys(deployUnits).forEach(name => {
            const networkDefaults = deployUnits[name]?.network_defaults;
            const podTemplateSpec =
               deployUnits[name].replica_set?.replica_set_template?.pod_template_spec ??
               deployUnits[name].multi_cluster_replica_set?.replica_set?.pod_template_spec;

            const ip6AddressRequests = podTemplateSpec?.spec?.ip6_address_requests;

            // const { network_id, virtual_service_ids, ip4_address_pool_id } = deployUnits[name]?.network_defaults ?? {};

            let customIp6AddressRequests = true;

            if (ip6AddressRequests === undefined) {
               customIp6AddressRequests = false;
            }

            // проверяем на дефолтные настройки, которые проставлял UI в старой/новой форме (DEPLOY-3850)
            // сейчас backend их сам подставляет с актуальной networkId, если поля ip6_address_requests нет
            if (ip6AddressRequests?.length === 2) {
               const backbone = ip6AddressRequests.find(v =>
                  isEqual(v, {
                     'enable_dns': true,
                     'network_id': networkDefaults?.network_id,
                     'vlan_id': 'backbone',
                  } as TPodSpec_TIP6AddressRequest),
               );

               const fastbone = ip6AddressRequests.find(v =>
                  isEqual(v, {
                     'enable_dns': true,
                     'network_id': networkDefaults?.network_id,
                     'vlan_id': 'fastbone',
                  } as TPodSpec_TIP6AddressRequest),
               );

               if (backbone && fastbone) {
                  customIp6AddressRequests = false;
               }
            }

            // if (networkDefaults?.ip4_address_pool_id) {
            // if (networkDefaults?.virtual_service_ids) {
            // if (networkDefaults?.ip4_address_pool_id || networkDefaults?.virtual_service_ids) {
            // if (networkDefaults && ip6AddressRequests?.find(s => s.network_id !== networkDefaults?.network_id)) {
            // 18
            // if (!networkDefaults) {  // 1 // dimaretunskiy-stage
            if (customIp6AddressRequests) {
               // 173
               // if (customIp6AddressRequests && ip6AddressRequests?.length !== 2) { // 6
               // if (ip6AddressRequests) { // 3869
               // if (ip6AddressRequests && networkDefaults) {
               // if (ip6AddressRequests && (networkDefaults?.ip4_address_pool_id || networkDefaults?.virtual_service_ids)) { // таких нет
               // if (ip6AddressRequests && ip6AddressRequests.length !== 2) {
               // console.log('networkId')
               // console.log(networkDefaults.network_id)
               // console.log(stage.meta?.id);
               // console.log('networkDefaults');
               // console.log(networkDefaults);
               // console.log('ip6AddressRequests');
               // console.log(ip6AddressRequests);
               result.push(stage.meta?.id ?? '');
            }
         });
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research disks by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const deployUnits = stage.spec?.deploy_units ?? {};

         Object.keys(deployUnits).forEach(name => {
            const podTemplateSpec =
               deployUnits[name].replica_set?.replica_set_template?.pod_template_spec?.spec ??
               deployUnits[name].multi_cluster_replica_set?.replica_set?.pod_template_spec?.spec;

            const diskVolumeRequests = podTemplateSpec?.disk_volume_requests;

            if (diskVolumeRequests?.length && diskVolumeRequests?.length > 2) {
               // console.log(stage.meta?.id)
               // console.log(diskVolumeRequests)
               // console.log(diskVolumeRequests?.length)
               result.push(stage.meta?.id ?? '');
            }
         });
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research volumes by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const deployUnits = stage.spec?.deploy_units ?? {};

         Object.keys(deployUnits).forEach(name => {
            const podAgentPayload =
               deployUnits[name].replica_set?.replica_set_template?.pod_template_spec?.spec?.pod_agent_payload ??
               deployUnits[name].multi_cluster_replica_set?.replica_set?.pod_template_spec?.spec?.pod_agent_payload;

            const volumes = podAgentPayload?.spec?.volumes;

            if (volumes?.length && volumes?.length > 1) {
               // console.log(stage.meta?.id)
               // console.log(volumes)
               // console.log(volumes?.length)
               result.push(stage.meta?.id ?? '');
            }
         });
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research deploy unit types by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const { deployUnits } = model;

         deployUnits.forEach(deployUnit => {
            if (deployUnit.type === DeployUnitType.MultiCluster) {
               // result.push(stage.meta?.id ?? '');

               if (Object.keys(deployUnit.locations).find(key => deployUnit.locations[key].podCount > 10)) {
                  result.push(stage.meta?.id ?? '');
               }
            }

            if (deployUnit.type === DeployUnitType.PerCluster) {
               // result.push(stage.meta?.id ?? '');
            }
         });
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research deploy units by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const { deployUnits } = model;

         // const tvmClientByLength = deployUnits
         //    .flatMap(v => v.tvm)
         //    .filter(v => v.enabled)
         //    .find(v => v.clients.length > 1);
         // if (tvmClientByLength) {
         //    // console.log(tvmClientByLength);
         //    result.push(stage.meta?.id ?? '');
         // }

         if (deployUnits.length > 5) {
            console.log(deployUnits.length);
            result.push(stage.meta?.id ?? '');
         }
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research endpoint sets by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const { deployUnits } = model;

         const endpointSets = deployUnits.flatMap(v => v.endpointSets);

         if (endpointSets.length > 40) {
            console.log(endpointSets.length);
            result.push(stage.meta?.id ?? '');
         }
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research simple stages', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const { deployUnits } = model;

         if (deployUnits.length === 1) {
            const { boxes } = deployUnits[0];

            if (boxes.length === 1) {
               const { workloads } = boxes[0];

               if (workloads.length === 1) {
                  result.push(stage.meta?.id ?? '');
               }
            }
         }
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Research box ids types by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         const { deployUnits } = model;

         deployUnits.forEach(deployUnit => {
            const boxes = deployUnit.boxes.map(box => box.id.toLowerCase().replace(/_/g, '-'));

            if (
               new Set(boxes).size !== boxes.length ||
               boxes.length !== boxes.filter(box => !/[.:@]/.test(box)).length
            ) {
               console.log(boxes);
               result.push(stage.meta?.id ?? '');
            }
         });
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Box guarantees by stage #DEPLOY-4755', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const deployUnits = stage.spec?.deploy_units ?? {};

         Object.keys(deployUnits).forEach(name => {
            const podTemplateSpec =
               deployUnits[name].replica_set?.replica_set_template?.pod_template_spec ??
               deployUnits[name].multi_cluster_replica_set?.replica_set?.pod_template_spec;

            const boxes = podTemplateSpec?.spec?.pod_agent_payload?.spec?.boxes;

            // боксы с гарантиями
            // const exist = boxes?.find(
            //    b => b.compute_resources?.memory_guarantee || b.compute_resources?.vcpu_guarantee,
            // );

            // боксы с гарантиями отличными от лимитов (или без лимитов)
            const exist = boxes?.find(
               b =>
                  (b.compute_resources?.memory_guarantee &&
                     b.compute_resources?.memory_guarantee !== b.compute_resources?.memory_limit) ||
                  (b.compute_resources?.vcpu_guarantee &&
                     b.compute_resources?.vcpu_guarantee !== b.compute_resources?.vcpu_limit),
            );

            if (exist) {
               console.log(exist.compute_resources);
               result.push(stage.meta?.id ?? '');
            }
         });
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Docker validation', () => {
      const result: string[] = [];
      const names: string[] = [];
      const tags: string[] = [];

      stagesMap.forEach(stage => {
         const deployUnits = stage.spec?.deploy_units ?? {};

         Object.keys(deployUnits).forEach(duName => {
            const imagesForBoxes = deployUnits[duName].images_for_boxes;

            if (imagesForBoxes) {
               Object.keys(imagesForBoxes).forEach(boxName => {
                  const invalidNameComponents = imagesForBoxes[boxName].name
                     .split('/')
                     .map(v => dockerNameComponentRegEx.test(v))
                     .filter(v => !v);

                  if (!isEmpty(invalidNameComponents)) {
                     result.push(stage.meta?.id ?? '');
                     names.push(`'${imagesForBoxes[boxName].name}'`);
                     // console.log(imagesForBoxes[boxName].name);
                     // console.log(stage.meta?.id ?? '');
                  }

                  if (!dockerTagRegEx.test(imagesForBoxes[boxName].tag)) {
                     result.push(stage.meta?.id ?? '');
                     tags.push(`'${imagesForBoxes[boxName].tag}'`);
                     // console.log(imagesForBoxes[boxName].tag);
                  }
               });
            }
         });
      });

      console.log(unique(result).join(', '));
      console.log('\n');
      console.log(`Total stages: ${unique(result).length}`);
      console.log(
         unique(names)
            .sort((a, b) => sortHandler(a.length, b.length))
            .join(', '),
      );
      console.log('\n');
      console.log(`Total names: ${unique(names).length}`);
      console.log('\n');
      console.log(
         unique(tags)
            .sort((a, b) => sortHandler(a.length, b.length))
            .join(', '),
      );
      console.log('\n');
      console.log(`Total tags: ${unique(tags).length}`);
   });

   it('Research tmp account by stage', () => {
      const result: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);

         if (model.project?.accountId === 'tmp') {
            console.log(model.project);
            result.push(stage.meta?.id ?? '');
         }
      });

      console.log(result.join('\n'));
      console.log('\n');
      console.log(`Total stages: ${result.length}`);
   });

   it('Stages with new secrets', () => {
      const stages: string[] = [];

      stagesMap.forEach(stage => {
         const model = StageConverter.fromApi(stage);
         for (const du of model.deployUnits) {
            const podTemplateSpec = DeployUnitConverter.getPodTemplateSpec(stage.spec!.deploy_units[du.id]);
            if (podTemplateSpec?.spec?.secret_refs) {
               stages.push(`${model.id}:${du.id}`);
            }
         }
      });

      console.log(stages);
   });

   it('Stages with long descriptions', () => {
      const result: {
         stage: string;
         description: string;
      }[] = [];

      stagesMap.forEach(stage => {
         const description = stage.spec?.revision_info?.description ?? '';

         if (description.length > 1000) {
            result.push({ stage: stage.meta?.id ?? '', description });
         }
      });

      console.log(result.map(v => v.stage));
      // console.log(result.map(v => v.description.length));
      // console.log(result.sort((a, b) => sortHandler(a.description.length, b.description.length)).map(v => v.stage));
      // console.log(result.sort((a, b) => sortHandler(a.description.length, b.description.length)).map(v => v.description.length));
   });

   describe('Great researches', () => {
      it('Deploy units', () => {
         const result = {
            total: 0,
            rs: { total: 0 },
            mcrs: { total: 0 },
            disks: { total: 0, '2+': { total: 0, stages: new Set() as Set<string> } },
            volumes: { total: 0, layers: { total: 0, stages: new Set() as Set<string> } },
            layers: { total: 0 },
            staticResources: { total: 0, verificationDisabled: { total: 0, stages: new Set() as Set<string> } },
            dynamicResources: { total: 0 },
            dockers: { total: 0, hashTags: { total: 0, stages: new Set() as Set<string> } },
            tvm: { total: 0, clients: { total: 0 } },
            patchersRevision: {
               total: 0,
               empty: {
                  total: 0,
                  stages: new Set() as Set<string>,
                  emptyLabels: {
                     autoupdate: { total: 0, stages: new Set() as Set<string> },
                     target: { total: 0, stages: new Set() as Set<string> },
                  },
               },

               // https://st.yandex-team.ru/DEPLOY-5195#6222258537bf95501da1f2d1
               invalid_du_patchers_autoupdate_revision_label: { total: 0, stages: new Set() as Set<string> },
               invalid_du_patchers_target_revision: { total: 0, stages: new Set() as Set<string> },
            },
            secrets: {
               old: { total: 0 },
               new: { total: 0 },
            },
            boxes: { total: 0 },
            workloads: { total: 0 },

            layersAndStaticResources: { deployUnits: 0, boxes: 0 },
         };

         const stageProjects = {
            accountId: {
               total: 0,
               stages: new Set() as Set<string>,
               empty: { total: 0, stages: new Set() as Set<string> },
            },
         };

         stagesMap.forEach(stage => {
            const stageName = stage.meta?.id ?? '';
            const deployUnits = stage.spec?.deploy_units ?? {};

            result.dynamicResources.total += Object.keys(stage.spec?.dynamic_resources ?? {}).length;

            result.total += Object.keys(deployUnits).length;

            if (stage.meta?.account_id && !isEmpty(stage.meta.account_id)) {
               stageProjects.accountId.total += 1;
               stageProjects.accountId.stages.add(stageName);
            } else {
               stageProjects.accountId.empty.total += 1;
               stageProjects.accountId.empty.stages.add(stageName);
            }

            Object.keys(deployUnits).forEach(deployUnitName => {
               const deployUnit = deployUnits[deployUnitName];

               if (deployUnit.replica_set) {
                  result.rs.total += 1;
               } else if (deployUnit.multi_cluster_replica_set) {
                  result.mcrs.total += 1;
               }

               const podTemplateSpec =
                  deployUnit.replica_set?.replica_set_template?.pod_template_spec?.spec ??
                  deployUnit.multi_cluster_replica_set?.replica_set?.pod_template_spec?.spec;

               const podAgentPayload = podTemplateSpec?.pod_agent_payload;

               const countDisks = (podTemplateSpec?.disk_volume_requests ?? []).length;

               result.disks.total += countDisks;

               if (countDisks > 1) {
                  result.disks['2+'].total += 1;
                  result.disks['2+'].stages.add(stageName);
               }

               result.volumes.total += (podAgentPayload?.spec?.volumes ?? []).length;

               const countLayers = (podAgentPayload?.spec?.resources?.layers ?? []).length;
               result.layers.total += countLayers;

               const countStaticResources = (podAgentPayload?.spec?.resources?.static_resources ?? []).length;
               result.staticResources.total += countStaticResources;

               if (countStaticResources > 0) {
                  const verificationDisabled = (podAgentPayload?.spec?.resources?.static_resources ?? []).filter(
                     v => v.verification?.disabled === true,
                  );

                  if (verificationDisabled.length > 0) {
                     result.staticResources.verificationDisabled.total += verificationDisabled.length;
                     result.staticResources.verificationDisabled.stages.add(stageName);
                  }
               }

               if (countLayers > 0 && countStaticResources > 0) {
                  const boxes = podAgentPayload?.spec?.boxes?.filter(
                     v => !isEmpty(v.rootfs?.layer_refs ?? []) && !isEmpty(v.static_resources ?? []),
                  );

                  if (!isEmpty(boxes)) {
                     result.layersAndStaticResources.boxes += boxes.length;
                  }

                  result.layersAndStaticResources.deployUnits += 1;
               }

               const countVolumeLayers = (
                  podAgentPayload?.spec?.volumes?.flatMap(v => v.generic?.layer_refs ?? []) ?? []
               ).length;

               if (countVolumeLayers) {
                  result.volumes.layers.stages.add(stageName);

                  podAgentPayload?.spec?.volumes?.forEach(volume => {
                     const layers = volume.generic?.layer_refs ?? [];
                     result.volumes.layers.total += layers.length;
                  });
               }

               result.dockers.total += (Object.keys(deployUnit.images_for_boxes ?? {}) ?? []).length;
               (Object.keys(deployUnit.images_for_boxes ?? {}) ?? [])
                  .filter(key => /sha256:/.test(deployUnit.images_for_boxes[key].tag))
                  .forEach(v => {
                     result.dockers.hashTags.total += 1;
                     result.dockers.hashTags.stages.add(stageName);
                  });

               result.secrets.old.total += (Object.keys(podTemplateSpec?.secrets ?? {}) ?? []).length;
               result.secrets.new.total += (Object.keys(podTemplateSpec?.secret_refs ?? {}) ?? []).length;

               result.patchersRevision.total += 1;
               if (!deployUnit.patchers_revision || isEmpty(deployUnit.patchers_revision)) {
                  result.patchersRevision.empty.total += 1;
                  result.patchersRevision.empty.stages.add(stageName);

                  if (!stage.labels?.du_patchers_autoupdate_revision?.[deployUnitName]) {
                     result.patchersRevision.empty.emptyLabels.autoupdate.total += 1;
                     result.patchersRevision.empty.emptyLabels.autoupdate.stages.add(stageName);
                  }

                  if (!stage.labels?.du_patchers_target_revision?.[deployUnitName]) {
                     result.patchersRevision.empty.emptyLabels.target.total += 1;
                     result.patchersRevision.empty.emptyLabels.target.stages.add(stageName);
                  }
               }

               if (deployUnit.tvm_config?.mode === TTvmConfig_EMode.ENABLED) {
                  if (!isEmpty(deployUnit.tvm_config.clients ?? [])) {
                     result.tvm.total += 1;
                     result.tvm.clients.total += deployUnit.tvm_config.clients.length;
                  }
               }

               result.boxes.total += (podAgentPayload?.spec?.boxes ?? []).length;
               result.workloads.total += (podAgentPayload?.spec?.workloads ?? []).length;

               if (stage.labels?.du_patchers_autoupdate_revision?.[deployUnitName]) {
                  if (deployUnit.patchers_revision === 11) {
                     // https://st.yandex-team.ru/DEPLOY-5195#6222258537bf95501da1f2d1
                     result.patchersRevision.invalid_du_patchers_autoupdate_revision_label.total += 1;
                     result.patchersRevision.invalid_du_patchers_autoupdate_revision_label.stages.add(stageName);
                  }
               }

               if (stage.labels?.du_patchers_target_revision?.[deployUnitName]) {
                  if (deployUnit.patchers_revision === 11) {
                     // https://st.yandex-team.ru/DEPLOY-5195#6222258537bf95501da1f2d1
                     result.patchersRevision.invalid_du_patchers_target_revision.total += 1;
                     result.patchersRevision.invalid_du_patchers_target_revision.stages.add(stageName);
                  }
               }
            });
         });

         console.log(`Total stages: ${stagesMap.size}`);
         console.log('Deploy units data:');
         console.log(result);
         // console.log(stageProjects.accountId.empty.stages);
         // console.log(stageProjects.accountId.empty.stages.size);
         console.log(result.patchersRevision.empty.emptyLabels.autoupdate.stages);
         console.log(result.patchersRevision.empty.emptyLabels.autoupdate.stages.size);
         // console.log(result.patchersRevision.empty.emptyLabels.target.stages);
         // console.log(result.patchersRevision.empty.emptyLabels.target.stages.size);
         // console.log(stageProjects.accountId.empty.stages);
         // console.log(stageProjects.accountId.empty.stages.size);
         // console.log(result.staticResources.verificationDisabled.stages);
         // console.log(result.staticResources.verificationDisabled.stages.size);
         // console.log(result.patchersRevision.invalid_du_patchers_target_revision.stages);
         // console.log(result.patchersRevision.invalid_du_patchers_target_revision.stages.size);
         // console.log(result.patchersRevision.invalid_du_patchers_autoupdate_revision_label.stages);
         // console.log(result.patchersRevision.invalid_du_patchers_autoupdate_revision_label.stages.size);
         // console.log(result.disks['2+'].stages);
         // console.log(result.patchersRevision.empty.stages.size);
         // console.log(result.dockers.hashTags.stages);
         // console.log(result.dockers.hashTags.stages.size);
      });
   });
});
