import { action } from '@storybook/addon-actions';
import { boolean, number, radios, select, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { json } from '@yandex-infracloud-ui/libs';
import {
   CheckBox,
   Icon,
   IconSize,
   RadioButton,
   Select,
   Spin,
   SpinSize,
   TextInput,
   Ticker,
   Tooltip,
   Tumbler,
   User,
} from 'lego-on-react';
import React, { useRef, useState } from 'react';

import legoErrors from './_styles/lego-errors.module.css';
import { LegoCheckBox, LegoRadioButton, LegoTextArea, LegoTextInput, LegoTumbler } from './lego-wrapers';

function TooltipStory() {
   // hooks
   const ref = useRef<HTMLButtonElement>();
   const [visible, setVisible] = useState(false);

   // handlers
   const hide = () => setVisible(false);
   const show = () => setVisible(true);
   const setRef = (node: HTMLButtonElement) => {
      ref.current = node;
   };

   // render
   return (
      <>
         <button ref={setRef} onMouseEnter={show} onMouseLeave={hide} type={'button'}>
            hover me!
         </button>

         <Tooltip anchor={() => ref.current} size={'m'} theme={'white'} visible={visible}>
            Super mega information
         </Tooltip>
      </>
   );
}

function TextInputStory() {
   // hooks
   const [v, setV] = useState('init');

   // handlers
   const update = (t: string) => setV(t);
   const nativeUpdate = (e: React.ChangeEvent<HTMLInputElement>) => setV(e.currentTarget.value);

   // render
   return (
      <>
         <p>
            LEGO: <TextInput theme={'normal'} size={'s'} placeholder={'Placeholder'} text={v} onChange={update} />
         </p>

         <p>
            Native: <input type={'text'} value={v} placeholder={'Placeholder'} onChange={nativeUpdate} />
         </p>

         <pre>{json(v)}</pre>
      </>
   );
}

export const ErrorHighlights = () => (
   <>
      <section className={legoErrors.error}>
         <h2>Inside of wrapper</h2>
         <LegoTextInput placeholder={'Placeholder'} />

         <LegoTextArea placeholder={'Placeholder'} rows={3} />
         <hr />
         <LegoCheckBox checked={false}>Checkbox false</LegoCheckBox>
         <LegoCheckBox theme={'normal'} size={'s'} checked={true}>
            Checkbox true
         </LegoCheckBox>
         <hr />
         <LegoTumbler size={'s'} theme={'normal'}>
            <Tumbler.Off id={''} side={'right'} text={'Tumbler label'} />
         </LegoTumbler>
         <hr />
         <LegoRadioButton value={radios('value', { x: 'x', y: 'y' }, 'x')} onChange={action('onChange')}>
            <RadioButton.Radio value={'x'}>
               <span style={{ color: 'red' }}>xx</span>
            </RadioButton.Radio>
            <RadioButton.Radio value={'y'}>yy</RadioButton.Radio>
         </LegoRadioButton>
      </section>

      <section>
         <h2>Direct</h2>

         <LegoTextInput theme={'normal'} size={'s'} placeholder={'Placeholder'} cls={legoErrors.error} />

         <LegoTextArea theme={'normal'} size={'s'} placeholder={'Placeholder'} cls={legoErrors.error} rows={3} />
         <hr />
         <CheckBox theme={'normal'} size={'s'} checked={false} cls={legoErrors.error}>
            Checkbox false
         </CheckBox>
         <LegoCheckBox theme={'normal'} size={'s'} checked={true} cls={legoErrors.error}>
            Checkbox true
         </LegoCheckBox>
         <hr />
         <LegoTumbler size={'s'} theme={'normal'} cls={legoErrors.error}>
            <Tumbler.Off id={''} side={'right'} text={'Tumbler label'} />
         </LegoTumbler>
         <hr />
         <LegoRadioButton
            theme={'normal'}
            size={'s'}
            view={'default'}
            tone={'grey'}
            cls={legoErrors.error}
            value={radios('value', { x: 'x', y: 'y' }, 'x')}
            onChange={action('onChange')}
         >
            <RadioButton.Radio value={'x'}>
               <span style={{ color: 'red' }}>xx</span>
            </RadioButton.Radio>
            <RadioButton.Radio value={'y'}>yy</RadioButton.Radio>
         </LegoRadioButton>
      </section>
   </>
);

export const IconCustomizable = () => (
   <Icon
      style={{ width: '40px' }}
      type={select('type', ['arrow', 'filter', 'close', 'cross', 'cross-websearch'], 'arrow')}
      direction={select('direction', ['left', 'top', 'right', 'bottom'], 'left')}
      size={select('size', ['ns', 'xs', 's', 'm', 'n', 'l', 'head'] as IconSize[], 's')}
      glyph={select(
         'glyph',
         ['carets-v', 'type-arrow', 'type-check', 'type-close', 'type-cross', 'type-filter', 'type-tick', 'x-sign'],
         '',
      )}
   />
);

export const UserEmpty = () => <User logoutOrigin={''} />;

export const UserFull = () => {
   const user = {
      id: 'khoden',
      login: 'khoden',
      name: 'Artem Berezin',
   };

   return (
      <User
         uid={user.id}
         yu={user.id}
         logoutOrigin={''}
         provider={'yandex-team'}
         retpath={'https://yandex-team.ru'}
         avatarHost={'https://center.yandex-team.ru'}
         passportHost={'https://passport.yandex-team.ru'}
         name={user.name}
         avatarId={user.login}
         addAccountHidden={true}
      />
   );
};

export const SelectSimple = () => (
   <Select
      theme={'normal'}
      size={'s'}
      type={'radio'}
      val={select('val', ['item1', 'item2', 'item3', 'item4'], 'item1')}
      onChange={action('onChange')}
      text={'vary'}
   >
      <Select.Item val={'item1'}>item1</Select.Item>
      <Select.Item val={'item2'}>item2</Select.Item>
      <Select.Item val={'item3'}>item3</Select.Item>
      <Select.Item val={'item4'}>item4 with long long long long long label</Select.Item>
   </Select>
);

export const SpinCustomizable = () => (
   <Spin size={select('size', ['xxs', 'xs', 's', 'm', 'l'] as SpinSize[], 's')} progress={boolean('progress', true)} />
);

export const TextAreaSimple = () => (
   <LegoTextArea placeholder={text('placeholder', 'Placeholder')} rows={number('rows', 3)} />
);

TextAreaSimple.state = { name: 'TextArea Simple' };

export const TextInputExample = () => <TextInputStory />;

export const TickerSimple = () => <Ticker count={number('count', 1)} />;

export const TumblerSimple = () => (
   <Tumbler size={'s'} theme={'normal'} checked={boolean('checked', false)} onChange={action('onChange')}>
      <Tumbler.Off id={''} side={'right'} text={text('Tumbler.Off: text', 'Подпись к флагу')} />
   </Tumbler>
);

export const TooltipExample = () => <TooltipStory />;

export default {
   title: 'other/lego-on-react 3',
} as Meta;
