import React, { ReactNode, useCallback, useEffect, useMemo, useRef, useState } from 'react';

import { classNames } from '@yandex-infracloud-ui/libs';
import { faChevronRight } from '@fortawesome/free-solid-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';

import classes from './CollapsibleContainer.module.css';

const DURATION = 200;

interface Props {
   title?: ReactNode;
   subtitle?: ReactNode;
   isOpened?: boolean;
   showArrow?: boolean;
   highlightTitle?: boolean;
   onOpen?(isOpened: boolean): void;
   className?: string;
   tabIndex?: number;
   openOnTitleClick?: boolean;
   openOnSubtitleClick?: boolean;
}

export const CollapsibleContainer: React.FC<Props> = ({
   title,
   subtitle,
   isOpened: isOpenedProp,
   showArrow,
   highlightTitle,
   className,
   children,
   tabIndex,
   openOnTitleClick = true,
   openOnSubtitleClick = true,
   onOpen,
}) => {
   const [isOpened, setOpened] = useState(isOpenedProp ?? false);
   const innerRef = useRef<HTMLDivElement>(null);
   const outerRef = useRef<HTMLDivElement>(null);

   const startTimestamp = useRef<number>();
   const stopHeight = useRef<number>();
   const startHeight = useRef<number>();

   const icon = useMemo(() => {
      if (!showArrow) {
         return null;
      }

      return <FontAwesomeIcon icon={faChevronRight} />;
   }, [showArrow]);

   const animate = useCallback((timestamp: number) => {
      if (stopHeight.current !== undefined && startHeight.current !== undefined && outerRef.current) {
         if (!startTimestamp.current) {
            startTimestamp.current = timestamp;
         }

         const deltaTime = timestamp - startTimestamp.current!;
         const deltaHeight = ((stopHeight.current - startHeight.current) * deltaTime) / DURATION;
         const newHeight = Math.floor(startHeight.current + deltaHeight);

         outerRef.current.style.height = `${newHeight}px`;

         if (deltaTime < DURATION) {
            requestAnimationFrame(animate);
         } else {
            outerRef.current.style.height = '';
            startTimestamp.current = undefined;
         }
      }
   }, []);

   const animateCallback = useCallback(
      (isOpening: boolean) => {
         const element = innerRef.current;
         if (element) {
            const { height } = element.getBoundingClientRect();
            if (isOpening) {
               stopHeight.current = height;
               startHeight.current = 0;
            } else {
               stopHeight.current = 0;
               startHeight.current = height;
            }

            requestAnimationFrame(animate);
         }
      },
      [animate],
   );

   const changeOpened = useCallback(
      (newOpenedState: boolean, triggerEvent: boolean) => {
         setOpened(newOpenedState);

         animateCallback(newOpenedState);

         if (triggerEvent && onOpen) {
            onOpen(newOpenedState);
         }
      },
      [animateCallback, onOpen],
   );

   const onClick = useCallback(() => {
      changeOpened(!isOpened, true);
   }, [isOpened, changeOpened]);

   useEffect(() => {
      changeOpened(isOpenedProp ?? false, false);
   }, [isOpenedProp]); //eslint-disable-line

   return (
      <div className={classNames(classes.container, className)}>
         <div
            role={'button'}
            tabIndex={tabIndex}
            className={classNames(classes.title, {
               [classes.highlightTitle]: highlightTitle,
               [classes.titleOpened]: isOpened,
            })}
            onClick={openOnTitleClick ? onClick : undefined}
         >
            {showArrow ? (
               <span className={classNames(classes.icon, { [classes.iconOpened]: isOpened })}>{icon}</span>
            ) : null}
            <span className={classes.titleContent}>{title}</span>
         </div>
         {subtitle ? (
            <div
               role={'button'}
               tabIndex={tabIndex}
               className={classes.subtitle}
               onClick={openOnSubtitleClick ? onClick : undefined}
            >
               {subtitle}
            </div>
         ) : null}
         <div className={classNames(classes.contentOuter, { [classes.outerOpened]: isOpened })} ref={outerRef}>
            <div className={classes.contentInner} ref={innerRef}>
               {children}
            </div>
         </div>
      </div>
   );
};
