import React from 'react';
import { Popover } from '@yandex-cloud/uikit';

import { useStartrekTicket } from '../../../redux';
import { handleStopPropagation } from '../../../utils';
import { EXTERNAL_LINKS } from '../../../models';

import { RowTicket } from './RowTicket';

import classes from './StartrekTicket.module.css';

export enum StartrekRenderMode {
   HOVER_TOOLTIP = 'hover',
   ROW = 'row',
}

export type StartrekProps = {
   id: string;
   mode?: StartrekRenderMode;
   className?: string;
   style?: React.CSSProperties;
   lazyLoad?: boolean; // загрузка данных потом, сам компонент ждет обновление из стора
};

export const startrekLoader = {
   hook: useStartrekTicket,
};

const TicketHref: React.FC<{ id: string }> = React.memo(({ id }) => (
   <a
      href={EXTERNAL_LINKS.startrekTicket(id)}
      onClick={handleStopPropagation}
      target={'_blank'}
      rel={'noreferrer noopener'}
   >
      {id}
   </a>
));

/**
 * id - номер задачи в трекере
 * mode - режим отображения:
 *         * ROW:   отображение в строку информацию с тикета
 *         * HOVER: отображение во всплывающем тултипе
 * className - класс для стилизации отображение блока в тултипе/строке
 */
export const StartrekTicket: React.FC<StartrekProps> = React.memo(
   ({ id, mode = StartrekRenderMode.ROW, className: rowTicketClassName = '', lazyLoad = false, style = {} }) => {
      const { ticket } = startrekLoader.hook(id, lazyLoad);
      const cls = [mode === StartrekRenderMode.HOVER_TOOLTIP ? classes.startrekTooltipMod : '', rowTicketClassName];

      return mode === StartrekRenderMode.ROW ? (
         <div style={style} className={cls.join(' ')}>
            <TicketHref id={id} />
            <RowTicket ticket={ticket} />
         </div>
      ) : (
         <Popover
            content={
               <div style={style} className={cls.join(' ')}>
                  <RowTicket ticket={ticket} />
               </div>
            }
         >
            <TicketHref id={id} />
         </Popover>
      );
   },
);

StartrekTicket.displayName = 'StartrekTicket';
