/* eslint-disable no-underscore-dangle */
import { Select } from 'lego-on-react';
import moment from 'moment-timezone';
import React from 'react';

// Российские таймзоны и значения их оффсетов на данный момент
const TIMEZONES = new Map<string, number>([
   ['Etc/UTC', 0],
   ['Europe/Kaliningrad', -120],
   ['Europe/Moscow', -180],
   ['Europe/Samara', -240],
   ['Asia/Yekaterinburg', -300],
   ['Asia/Omsk', -360],
   ['Asia/Krasnoyarsk', -420],
   ['Asia/Irkutsk', -480],
   ['Asia/Yakutsk', -540],
   ['Asia/Vladivostok', -600],
   ['Asia/Magadan', -660],
   ['Asia/Kamchatka', -720],
]);

interface Props {
   timeZoneKey?: string;

   onChange(timeZone: Timezone): void;
}

interface Timezone {
   key: string;
   title: string;
   offset: number;
}

/**
 * Возвращает список используемых в контроле таймзон
 *
 * @returns
 */
const getTimezones = () => {
   const currentTimezone = moment.tz.guess();
   const russianTimezones = moment.tz.zonesForCountry('RU');

   const Timezones = Array.from(TIMEZONES.keys()).map(timezoneKey => {
      const timezoneOffset = TIMEZONES.get(timezoneKey);
      return timezoneOffset === undefined
         ? undefined
         : {
              key: timezoneKey,
              title: timezoneOffset !== 0 ? `UTC+${Math.abs(timezoneOffset / 60)}` : 'UTC',
              offset: timezoneOffset,
           };
   }) as Timezone[];

   const isRussianTimezone = russianTimezones.includes(currentTimezone);

   if (!isRussianTimezone) {
      Timezones.push(createCurrentTimezone());
   }

   return Timezones;
};

/**
 * Создает новую таймзону на основании текущего оффсета и кода, который вернет moment-timezone
 *
 * @returns
 */
const createCurrentTimezone: () => Timezone = () => {
   const currentOffset = new Date().getTimezoneOffset();
   const currentTimezone = moment.tz.guess();

   let offsetTitle = '';
   if (currentOffset > 0) {
      offsetTitle = `-${Math.abs(currentOffset) / 60}`;
   } else if (currentOffset < 0) {
      offsetTitle = `+${Math.abs(currentOffset) / 60}`;
   }

   return {
      key: currentTimezone,
      title: `UTC${offsetTitle}`,
      offset: currentOffset,
   };
};

export class TimeZonePicker extends React.Component<Props> {
   private static _timezones: Timezone[] = getTimezones();

   /**
    * Возвращает таймзону на основании оффсета
    *
    * @static
    * @memberof TimeZonePicker
    */
   static GetCurrentTimezone: () => Timezone | undefined = () => {
      const currentOffset = new Date().getTimezoneOffset();

      const timezone = TimeZonePicker._timezones.find(x => x.offset === currentOffset);
      if (timezone) {
         return timezone;
      }

      const newTimezone = createCurrentTimezone();
      TimeZonePicker._timezones.push(newTimezone);

      return newTimezone;
   };

   /**
    * Возвращает таймзону по коду
    *
    * @static
    * @memberof TimeZonePicker
    */
   static GetTimezoneByKey: (key: string) => Timezone = key => {
      const timezone = TimeZonePicker._timezones.find(x => x.key === key);

      if (timezone) {
         return timezone;
      }

      const newTimezone = createCurrentTimezone();
      TimeZonePicker._timezones.push(newTimezone);

      return newTimezone;
   };

   onSelect: (newValue: string | string[]) => void = newValue => {
      const key = typeof newValue === 'string' ? newValue : newValue.pop();

      const timeZoneData = TimeZonePicker._timezones.find(x => x.key === key);

      if (timeZoneData) {
         this.props.onChange(timeZoneData);
      }
   };

   render() {
      const selectItems = TimeZonePicker._timezones.map(({ key, title }) => (
         <Select.Item key={key} val={key}>
            {title}
         </Select.Item>
      ));

      let value = this.props.timeZoneKey;
      if (!value) {
         const currentTimezone = TimeZonePicker.GetCurrentTimezone();
         value = currentTimezone?.key;
      }

      return (
         <Select theme={'normal'} size={'s'} type={'radio'} val={value} onChange={this.onSelect}>
            {selectItems}
         </Select>
      );
   }
}
