import { ClipboardButton, Label } from '@yandex-cloud/uikit';
import { Collapse } from '@yandex-data-ui/common';
import { classNames, formatDate, Json } from '@yandex-infracloud-ui/libs';
import React from 'react';

import { WarningLevelData } from '../../../models/ui/runtimeDeploy';
import { YtError } from '../YtError/YtError';

import classes from './YpError.module.css';

const tryJSON = (obj: string): { isJson: boolean; json: any } => {
   try {
      const json = JSON.parse(obj);
      return {
         isJson: true,
         json,
      };
   } catch (e) {
      return {
         isJson: false,
         json: null,
      };
   }
};

const Message: React.FC<{ text: string; className?: string }> = ({ text, children, className }) => (
   <div className={classNames(classes.message, className)}>
      {children}
      <div className={classes.copy}>
         <ClipboardButton text={text} size={16} />
      </div>
   </div>
);

const TerminalTail: React.FC<{ title: string; children: string }> = ({ title, children }) => (
   <div className={classes.additional}>
      <Collapse title={<strong className={classes.additionalHeader}>{title}</strong>} defaultIsExpand={true}>
         <Message text={children} className={classes.terminalText}>
            <pre>{children}</pre>
         </Message>
      </Collapse>
   </div>
);

const msTimeFormat = 'SSS';
const Time: React.FC<{ title: string; time: number }> = ({ time, title }) => {
   const date = new Date(time);
   const ms = date.getMilliseconds();
   return (
      <span className={classes.time}>
         {title}: {formatDate(time)}
         {ms !== 0 && <span className={classes.milliseconds}>:{formatDate(time, msTimeFormat)}</span>}
      </span>
   );
};

interface Props {
   level: WarningLevelData<string>;
}

export const YpError: React.FC<Props> = React.memo(({ level }) => (
   <div>
      {Object.keys(level).map(levelName => {
         const warnings = level[levelName];
         if (!warnings) {
            return null;
         }
         if (warnings.length === 0) {
            return null;
         }
         return (
            <div key={levelName}>
               <div className={classes.levelName}>{levelName}</div>
               {warnings.map(warning => {
                  const { error, failStatus, failed, message, meta } = warning;
                  const { isJson: isMessageJson, json: messageJson } = tryJSON(message ?? '');
                  const { failReason, terminalTails, times } = failStatus ?? {};
                  const { isJson: isFailJson, json: failJson } = tryJSON(failReason ?? '');
                  return (
                     <div className={classes.warning}>
                        {meta && (
                           <>
                              {meta.locations && meta.locations.size > 0 && (
                                 <div className={classes.locations}>
                                    {Array.from(meta.locations).map(location => (
                                       <Label key={location}>{location.toUpperCase()}</Label>
                                    ))}
                                 </div>
                              )}
                           </>
                        )}
                        {message && (
                           <Message text={message}>
                              {isMessageJson ? <Json obj={messageJson} /> : <pre>{message}</pre>}
                           </Message>
                        )}
                        {failReason && (
                           <>
                              {times &&
                                 Array.from(times).map(([name, time], i) => (
                                    <>
                                       {i !== 0 && ' — '}
                                       <Time title={`${name} time`} time={time} />
                                    </>
                                 ))}
                              <Message text={failReason}>
                                 {isFailJson ? <Json obj={failJson} /> : <pre>{failReason}</pre>}
                              </Message>
                              {Array.from(terminalTails?.entries() ?? []).map(([name, value]) => (
                                 <TerminalTail title={`${name} tail`}>{value}</TerminalTail>
                              ))}
                           </>
                        )}
                        {failed?.active && (
                           <>
                              {failed.reason && (
                                 <div>
                                    Reason: <span className={classes.errorReason}>{failed.reason}</span>
                                 </div>
                              )}
                              {failed.lastTransitionTime && (
                                 <Time title={'Last transition time'} time={failed.lastTransitionTime} />
                              )}
                              {failed.message && (
                                 <Message text={failed.message}>
                                    <pre>{failed.message.replace(/\r/g, '\n')}</pre>
                                 </Message>
                              )}
                              {failed?.rawJSON && (
                                 <Collapse title={<span>JSON</span>}>
                                    <Message text={failed.rawJSON}>
                                       <Json obj={JSON.parse(failed.rawJSON)} />
                                    </Message>
                                 </Collapse>
                              )}
                           </>
                        )}
                        {error && (
                           <div className={classes.message}>
                              <YtError error={error} />
                           </div>
                        )}
                     </div>
                  );
               })}
            </div>
         );
      })}
   </div>
));

YpError.displayName = 'YpError';
