import { camelCase } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useMemo } from 'react';

import { BaseLogPage, columnPresets, JsonCell, SmartTable } from '../logs-core';

import { LevelCell } from './cells/LevelCell/LevelCell';
import { LinkCell } from './cells/LinkCell/LinkCell';
import { LogSourceCell } from './cells/LogSourceCell/LogSourceCell';
import { MessageCell } from './cells/MessageCell/MessageCell';
import { clickableRequestFields, customRequestFields, requiredFilters } from './constants';
import { DeployUnitFilter } from './filters/DeployUnitFilter/DeployUnitFilter';
import { LevelFilter } from './filters/LevelFilter/LevelFilter';
import { ColumnNames } from './models/ColumnNames';

const ST = SmartTable;
const cp = columnPresets;

interface Props {
   apiEndpoint: string;
   deployUnitIds: string[];
   filtersClassName?: string;
   projectId: string;
   queryString: string;
   stageId: string;

   onQueryStringUpdate(qs: string): void;
}

// TODO: DEPLOY-5955
const snakeCase = true;
const formatName = (name: string) => (snakeCase ? name : camelCase(name));
const requestFields = clickableRequestFields.map(field => {
   field.name = formatName(field.name);

   return field;
});

export const DeployLogPage: React.FC<Props> = React.memo(
   ({ apiEndpoint, deployUnitIds, filtersClassName, projectId, queryString, stageId, onQueryStringUpdate }) => {
      const forcedFilters = useMemo(
         () => ({
            stageId,
            projectId,
         }),
         [stageId, projectId],
      );

      const filters = (
         <>
            <ST.Filter
               name={'deployUnitId'}
               defaultValue={undefined}
               component={DeployUnitFilter}
               internalProps={{ deployUnitIds }}
               getValueFromUrlParams={DeployUnitFilter.getValueFromUrlParams}
               setUrlParamsFromValue={DeployUnitFilter.setUrlParamsFromValue}
            />

            <ST.Filter
               name={formatName(ColumnNames.LogLevel)}
               defaultValue={undefined}
               component={LevelFilter}
               internalProps={{ filterName: formatName(ColumnNames.LogLevel) }}
               getValueFromUrlParams={LevelFilter.getValueFromUrlParams}
               setUrlParamsFromValue={LevelFilter.setUrlParamsFromValue}
            />
         </>
      );

      const columns = (
         <>
            <ST.Column
               name={formatName(ColumnNames.LogLevel)}
               header={'Level'}
               component={LevelCell}
               presets={cp.default}
            />
            <ST.Column name={ColumnNames.Message} component={MessageCell} presets={cp.default} />
            <ST.Column name={ColumnNames.Source} component={LogSourceCell} presets={cp.default} />
            <ST.Column name={formatName(ColumnNames.LoggerName)} header={'Logger'} presets={cp.default} />
            <ST.Column
               name={formatName(ColumnNames.LogLink)}
               header={'Link'}
               component={LinkCell}
               presets={cp.default}
            />
            <ST.Column name={formatName(ColumnNames.LogLevelInt)} header={'Log level Int'} presets={cp.details} />
            <ST.Column name={ColumnNames.Box} presets={cp.details} />
            <ST.Column name={ColumnNames.Workload} presets={cp.details} />
            <ST.Column name={ColumnNames.Context} presets={cp.ignored} component={JsonCell} />
            <ST.Column name={ColumnNames.Pod} presets={cp.details} />
            <ST.Column
               name={formatName(ColumnNames.PodTransientFqdn)}
               header={'Pod transient FQDN'}
               presets={cp.details}
            />
            <ST.Column
               name={formatName(ColumnNames.PodPersistentFqdn)}
               header={'Pod persistent FQDN'}
               presets={cp.details}
            />
            <ST.Column name={formatName(ColumnNames.ContainerId)} header={'PodPersistentFqdn'} presets={cp.details} />
            <ST.Column name={ColumnNames.Host} presets={cp.details} />
            <ST.Column name={formatName(ColumnNames.NodeFqdn)} header={'Node FQDN'} presets={cp.details} />
            <ST.Column name={formatName(ColumnNames.ThreadName)} header={'Thread name'} presets={cp.details} />
            <ST.Column name={formatName(ColumnNames.UserId)} header={'User Id'} presets={cp.details} />
            <ST.Column name={formatName(ColumnNames.RequestId)} header={'Request Id'} presets={cp.details} />
            <ST.Column name={formatName(ColumnNames.StackTrace)} header={'Stack trace'} presets={cp.details} />
            <ST.Column name={ColumnNames.Seq} presets={cp.details} />
         </>
      );

      return (
         <BaseLogPage
            apiEndpoint={apiEndpoint}
            columns={columns}
            filters={filters}
            forcedFilters={forcedFilters}
            requiredFilters={requiredFilters}
            // отдельные параметры могут быть не уникальными,
            // для уникального ключа нужны 3 параметра
            idColumns={[ColumnNames.Seq, ColumnNames.Timestamp, ColumnNames.ContainerId]}
            onQueryStringUpdate={onQueryStringUpdate}
            queryString={queryString}
            hideColumnPresets={true} // DEPLOY-5824
            filtersClassName={filtersClassName}
            requestFields={requestFields}
            customRequestFields={customRequestFields}
            snakeCase={snakeCase}
         />
      );
   },
);

DeployLogPage.displayName = 'DeployLogPage';

// {
//    "box": "box",
//    "stack_trace": "debug test stack trace (1627290280099)",
//    "workload": "WorkloadJsonLogs",
//    "log_level": "DEBUG",
//    "seq": "18446744073709513555",
//    "node_fqdn": "sas3-5602.search.yandex.net",
//    "thread_name": "",
//    "timestamp": "2021-11-24T11:02:50.313479Z",
//    "container_id": "pod_agent_box_box/workload_WorkloadJsonLogs_start",
//    "pod_transient_fqdn": "sas3-5602-2.w3yabiy2ygkfafcp.sas.yp-c.yandex.net",
//    "host": "sas3-5602-2.w3yabiy2ygkfafcp.sas.yp-c.yandex.net",
//    "pod_persistent_fqdn": "w3yabiy2ygkfafcp.sas.yp-c.yandex.net",
//    "logger_name": "debug",
//    "context": "{\"status\":{\"code\":200,\"message\":\"OK\"},\"context\":{\"minus-minus\":\"debug minus\",\"space space\":\"debug space\",\"context\":\"debug context\",\"dot.dot\":\"debug dot\"},\"level\":\"DEBUG\"}",
//    "request_id": "debug request",
//    "user_id": "",
//    "pod": "w3yabiy2ygkfafcp",
//    "message": "debug test message (1627290280099)",
//    "log_level_int": "0"
// }
