import { faCheck, faLink } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button, CopyToClipboard } from '@yandex-cloud/uikit';
import { ExternalLink, fromQuery } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useEffect, useMemo, useRef } from 'react';
import { useLocation } from 'react-router';

import { urlBuilder } from '../../../../models';
import { StageLogsParams } from '../../../../models/ui';
import { handleStopPropagation } from '../../../../utils';
import { CellProps, DataFilters, useLogPageContext } from '../../../logs-core';
import { currentIndexSymbol, currentTokenSymbol, LogEntry } from '../../../logs-core/log-page/api/models/LogEntry';
import { getDateUrlFormat } from '../../../logs-core/log-page/filters/IntervalFilter/IntervalFilter';
import { useSmartTableContext } from '../../../logs-core/smart-table/hooks/useSmartTableContext';

import classes from './LinkCell.module.css';

export const LinkCell: React.FC<CellProps<LogEntry>> = React.memo(({ rowEntry }) => {
   const { queryProvider } = useLogPageContext()!;
   const { stageId } = queryProvider.getForcedRequestParams();

   const { origin } = window.location;
   const { search } = useLocation();
   const { range, highlight } = fromQuery(search);

   const linkRef = useRef<HTMLDivElement>(null);

   const {
      filtersHook: { filters },
   } = useSmartTableContext();

   const link = useMemo(() => {
      const {
         deployUnitId,
         limit,
         order,
         query: { query },
         interval,
      } = filters;

      const params: StageLogsParams = {
         deployUnitId,
         from: getDateUrlFormat(interval?.from?.absolute),
         to: getDateUrlFormat(interval?.to?.absolute),
         highlight: rowEntry[currentIndexSymbol],
         range: rowEntry[currentTokenSymbol],
         limit,
         order,
         query,
      };

      return origin + urlBuilder.stageLogs(stageId, params);
   }, [filters, origin, rowEntry, stageId]);

   useEffect(() => {
      if (
         linkRef.current !== null &&
         range &&
         highlight &&
         rowEntry[currentTokenSymbol] === range &&
         rowEntry[currentIndexSymbol] === Number(highlight)
      ) {
         // верхняя позиция строки выбранного лога
         const rowTop = linkRef.current.closest('tr')?.getBoundingClientRect().top;

         if (rowTop) {
            // высота хидера и блока фильтров, которые загораживают контент под собой,
            // вычитаем их для более точного скролла к выбранной строке
            // https://st.yandex-team.ru/DEPLOY-5227#628670f19b318615413b6053
            const headerHeight = document.querySelector('header')?.getBoundingClientRect().height;
            const filtersHeight = document
               .querySelector(`[${DataFilters.name}="${DataFilters.value}"]`)
               ?.getBoundingClientRect().height;

            if (headerHeight && filtersHeight) {
               window.scrollTo({
                  top: rowTop - (headerHeight + filtersHeight) - 4,
                  left: 0,
                  behavior: 'smooth',
               });
            } else {
               // запасной, менее красивый скролл, на случай если что-то пошло не так на предыдущем шаге
               // (например, всё сломалось после переезда на новую навигацию)
               linkRef.current.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
         }
      }
   }, [range, highlight, linkRef, rowEntry]);

   return (
      <div ref={linkRef} className={classes.wrapper} onClick={handleStopPropagation} role={'button'} tabIndex={0}>
         <div className={classes.button} title={'Copy Link'}>
            <CopyToClipboard timeout={1000} text={link}>
               {state => (
                  <Button size={'s'} view={'flat'} className={classes.copy}>
                     <FontAwesomeIcon icon={state === 'success' ? faCheck : faLink} />
                  </Button>
               )}
            </CopyToClipboard>
         </div>

         <div className={classes.button} title={'Open Link in New Tab'}>
            <ExternalLink href={link} grey={true} />
         </div>
      </div>
   );
});

LinkCell.displayName = 'LinkCell';
