import { YCSelect } from '@yandex-data-ui/common';
import { isEmpty } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect, useMemo } from 'react';

import { FilterProps, getCleanExpressions, PlainObject, useSmartTableContext } from '../../../logs-core';

import classes from './LevelFilter.module.css';

const items = ['ERROR', 'WARNING', 'INFO', 'DEBUG'].map(id => ({
   title: id,
   value: id,
}));

const LevelFilterImpl: React.FC<FilterProps<string[] | undefined, { filterName: string }>> = React.memo(
   ({ internalProps, value, onUpdate }) => {
      const filterName = useMemo(() => internalProps?.filterName ?? [], [internalProps?.filterName]);

      const emptyValue = useMemo(() => isEmpty(value), [value]);

      const {
         filtersHook: {
            filters: {
               query: { query },
            },
         },
      } = useSmartTableContext();

      const hasCustomFilters = useMemo(() => getCleanExpressions(query).some(v => v.key === filterName), [
         filterName,
         query,
      ]);

      useEffect(() => {
         if (hasCustomFilters && !emptyValue) {
            onUpdate(undefined, true);
         }
      }, [hasCustomFilters, query, emptyValue, onUpdate]);

      const handleUpdate = useCallback(
         (v: string[]) => {
            onUpdate(isEmpty(v) ? undefined : v, true);
         },
         [onUpdate],
      );

      return (
         <div className={classes.wrapper}>
            <YCSelect
               value={value}
               items={items}
               onUpdate={handleUpdate}
               controlWidth={260}
               showSearch={false}
               disabled={hasCustomFilters}
               type={'multiple'}
               placeholder={'any level'}
            />
         </div>
      );
   },
);

export const LevelFilter = Object.assign(LevelFilterImpl, {
   getValueFromUrlParams(urlParams: PlainObject, filterName: string): string[] {
      const hasCustomFilters = getCleanExpressions(urlParams.query ?? '').some(v => v.key === filterName);

      if (hasCustomFilters) {
         return [];
      }

      return urlParams.level?.split(',');
   },
   setUrlParamsFromValue(v: string[]): PlainObject {
      const result: PlainObject = {};

      if (v) {
         result.level = v.join(',');
      }

      return result;
   },
});

LevelFilter.displayName = 'LevelFilter';
