import { Button } from '@yandex-cloud/uikit';
import {
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   IModalProps,
   Loader,
   ModalLayout,
} from '@yandex-infracloud-ui/libs';
import { Formik, FormikHelpers } from 'formik';
import React, { useCallback, useMemo, useRef, useState } from 'react';
import { useDispatch } from 'react-redux';

import { getApprovalsCount, isApprovalChanged, TicketAction } from '../../../models/ui';
import {
   doDeployTicketAction,
   useDeployTicket,
   useDeployTicketData,
   useNetworkErrors,
   useRequestControl,
} from '../../../redux';
import { YpErrorTooltip } from '../../network';
import {
   DeployTicketActionForm,
   deployTicketFormInitialValue,
   TicketActionParams,
   ticketActionParamsValidationSchema,
} from '../DeployTicketActionForm/DeployTicketActionForm';

import classes from '../../../design/commonModals.module.css';
import { TDeployTicketSpec_EDeployTicketSourceType } from '../../../proto-typings';

export interface TicketActionModalLabels {
   title: string;
   actionInProcess: string;
   submitButton: string;
}

interface Props extends IModalProps<TicketActionParams> {
   action: TicketAction;
   initialMessage: string; // подбираем удобное сообщение
   labels: TicketActionModalLabels;
   patchIds: string[];
   ticketId: string;
}

export const DeployTicketActionModal: React.FC<Props> = React.memo(
   ({ action, initialMessage, labels, ticketId, patchIds, ok, cancel }) => {
      const [actionStart, setActionStart] = useState(false);
      const requestKey = `ticket-action-${action}-${ticketId}`;

      const { deployTicket } = useDeployTicket(ticketId);
      const { approvalPermissions, approvalPolicySpec } = useDeployTicketData(ticketId);

      const patchList: string[] = useMemo(
         () => (patchIds.length > 0 ? patchIds : deployTicket?.patches?.map(patch => patch.id) ?? []),
         [deployTicket?.patches, patchIds],
      );

      const patchCaption = patchList.join(', ');

      const count = deployTicket
         ? getApprovalsCount({ ticket: deployTicket, approvalPermissions, approvalPolicySpec })
         : null;

      const approvalChanged = deployTicket && count && isApprovalChanged(deployTicket, count);

      const formHooks = useMemo(
         () =>
            ({
               onFieldChange: () => setActionStart(false),
            } as FormHooks<TicketActionParams>),
         [],
      );

      const initialValuePatched = useMemo(() => ({ ...deployTicketFormInitialValue, message: initialMessage }), [
         initialMessage,
      ]);

      const formRefs = useRef<{ values?: TicketActionParams; helpers?: FormikHelpers<TicketActionParams> }>({});

      const { activate } = useRequestControl([requestKey], {
         onSuccess: () => {
            ok(formRefs.current.values ?? initialValuePatched);
         },
         onError: () => {
            formRefs.current.helpers?.setSubmitting(false);
         },
      });

      const dispatch = useDispatch();
      const handleSubmit = useCallback(
         (values: TicketActionParams, helpers: FormikHelpers<TicketActionParams>) => {
            formRefs.current = {
               values,
               helpers,
            };
            setActionStart(true);
            activate(() => {
               dispatch(
                  doDeployTicketAction.withRequestKey(requestKey)(
                     action,
                     ticketId,
                     patchIds,
                     values.message,
                     values.reason,
                  ),
               );
            });
         },
         [activate, dispatch, requestKey, action, ticketId, patchIds],
      );

      const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

      const errors = useNetworkErrors([requestKey]);
      const error = errors[requestKey];
      const showError = actionStart && Boolean(error?.error);

      const mostLikelyApprovalChaged = showError && action === TicketAction.Commit && approvalChanged;

      return (
         <ModalLayout title={labels.title} showFooter={false} onDismiss={dismiss} data-e2e={'Ticket:ActionModal'}>
            <p>
               <strong>{ticketId}</strong>:{' '}
               {deployTicket?.sourceType === TDeployTicketSpec_EDeployTicketSourceType.RELEASE_INTEGRATION &&
               patchIds.length > 0 ? (
                  patchCaption
               ) : (
                  <span title={patchCaption} className={classes.allPatches}>
                     all patches
                  </span>
               )}
            </p>
            <Formik
               initialValues={initialValuePatched}
               onSubmit={handleSubmit}
               validationSchema={ticketActionParamsValidationSchema}
            >
               {form => (
                  <FormHooksContext.Provider value={formHooks}>
                     <DeployTicketActionForm action={action}>
                        <footer
                           className={classes.footer}
                           style={{
                              display: 'flex',
                              alignItems: 'center',
                              justifyContent: 'flex-end',
                              marginBottom: '8px',
                           }}
                        >
                           <div style={{ display: 'inline-block' }}>
                              {/* в TS 4.4 переменные должны учитываться */}
                              {showError && !form.isSubmitting && <YpErrorTooltip error={error!.error!} />}
                              {mostLikelyApprovalChaged && !form.isSubmitting && (
                                 <p>
                                    Most likely, the approval policy was changed. Approval statuses for deploy tickets
                                    are recalculated only for actions such as <strong>approve</strong> and{' '}
                                    <strong>disapprove</strong>
                                 </p>
                              )}
                           </div>

                           <Loader visible={form.isSubmitting} inline={true} text={labels.actionInProcess} />

                           <Button
                              size={'m'}
                              view={'flat'}
                              onClick={dismiss}
                              disabled={form.isSubmitting}
                              className={classes.button}
                           >
                              Cancel
                           </Button>

                           <Button
                              size={'m'}
                              view={'action'}
                              type={'submit'}
                              className={classes.button}
                              disabled={!form.isValid || form.isSubmitting}
                           >
                              {labels.submitButton}
                           </Button>
                        </footer>
                     </DeployTicketActionForm>
                  </FormHooksContext.Provider>
               )}
            </Formik>
         </ModalLayout>
      );
   },
);

DeployTicketActionModal.displayName = 'DeployTicketActionModal';
