import { Button } from '@yandex-cloud/uikit';
import { classNames, debounceFunc, fromQuery, isEqual, queryValueToSet } from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useEffect, useMemo, useRef, useState } from 'react';
import { useDispatch } from 'react-redux';
import { useHistory, useLocation } from 'react-router';

import { RuleSelect } from './components/RuleSelect/RuleSelect';
import { continuationTokensSlice } from '../../../redux';
import {
   ActiveTicketStatuses,
   TicketStatus,
   DeployTicketFilters,
   initialDeployTicketFilters,
   emptyDeployTicketFilters,
   ReleaseRuleIdsAndDrafts,
   OnlyTicketStatuses,
   draftLabel,
} from '../../../models/ui';
import classes from './DeployTicketFiltersView.module.css';
import { patchUrlQuery } from '../../../utils';
import { DeployTicketStatusSelect } from './components/DeployTicketStatusSelect/DeployTicketStatusSelect';
import { Column } from '../model';
import { IdTextInput } from './components/IdTextInput/IdTextInput';

type UrlFilters = {
   [k in keyof DeployTicketFilters]: string | undefined;
};

interface Props {
   filters: DeployTicketFilters;
   setFilters: (filters: DeployTicketFilters) => void;
   defaultFilters?: Partial<DeployTicketFilters>;
   className?: string;
   excludeColumns?: Set<Column>;
}

export const DeployTicketFiltersView: React.FC<Props> = React.memo(
   ({ className, filters: globalFilters, setFilters: setGlobalFilters, defaultFilters, excludeColumns }) => {
      const history = useHistory();
      const location = useLocation();
      const dispatch = useDispatch();

      const filterValues: DeployTicketFilters = useMemo(
         () => ({
            ...globalFilters,
            ...defaultFilters,
         }),
         [defaultFilters, globalFilters],
      );

      // локальные фильтры, которые отображают значения полей
      // применение фильтрации непосредственно к списку ТОЛЬКО через функцию runFiltration
      const [filters, setFilters] = useState(filterValues);

      const isUsingFilters = useMemo(() => !isEqual(filters, emptyDeployTicketFilters), [filters]);

      const { releaseRuleIdsAndDrafts, releaseId, stageId, ticketId, ticketStatuses } = filters;

      const locationRef = useRef(location);
      useEffect(() => {
         locationRef.current = location;
      }, [location]);

      const runFiltration = useMemo(
         () =>
            debounceFunc(() => {
               // задержка, чтобы url успел измениться перед ручным запуском фильтра
               const filtersFromUrl = parseFiltersFromUrl(locationRef.current.search);
               // location.search - source of truth для фильтрации
               setGlobalFilters(filtersFromUrl);
               dispatch(continuationTokensSlice.actions.resetTickets());
               // dispatch(releasesSlice.actions.setIsFiltering(true));
            }, 250),
         [dispatch, setGlobalFilters],
      );

      const beautifyFiltersForURL = useCallback((fullFilters: DeployTicketFilters) => {
         const cleanFilters = { ...fullFilters };
         if (
            cleanFilters.ticketStatuses.size === ActiveTicketStatuses.length &&
            ActiveTicketStatuses.every(status => cleanFilters.ticketStatuses.has(status))
         ) {
            cleanFilters.ticketStatuses = new Set();
         }

         return cleanFilters;
      }, []);

      const handleFilter = useCallback(() => {
         patchUrlQuery(history, locationRef.current, beautifyFiltersForURL(filters));
         runFiltration();
      }, [history, filters, runFiltration, beautifyFiltersForURL]);

      useEffect(() => {
         // фильтрация из URL при открытии страницы
         // похоже на костыль: даже если фильтры не указаны, запрос без фильтров инициируется отсюда. Но это самый красивый вариант из тех, что пробовал (usovdm@)
         runFiltration();
      }, [runFiltration]);

      // region handlers
      const filtersUpdater = useMemo(() => {
         const createFieldUpdater = <T extends keyof DeployTicketFilters>(field: T) => (
            value: DeployTicketFilters[T],
         ) => {
            const newFilters = { ...filters, [field]: value };
            setFilters(newFilters);
         };

         return (Object.keys(initialDeployTicketFilters) as (keyof DeployTicketFilters)[]).reduce((updater, name) => {
            updater[name] = createFieldUpdater(name);
            return updater;
         }, {} as { [K in keyof DeployTicketFilters]-?: (v: DeployTicketFilters[K]) => void });
      }, [filters, setFilters]);

      const handleResetFilters = useCallback(() => {
         patchUrlQuery(history, location, beautifyFiltersForURL(initialDeployTicketFilters));
         runFiltration();
         setFilters(filterValues);
      }, [history, location, beautifyFiltersForURL, runFiltration, filterValues]);
      // endregion

      const filterFields: Partial<Record<Column, () => ReactNode>> = {
         id: () => <IdTextInput value={ticketId} onChange={filtersUpdater.ticketId} title={'Ticket id'} />,
         stage: () => <IdTextInput value={stageId} onChange={filtersUpdater.stageId} title={'Stage id'} />,
         release: () => <IdTextInput value={releaseId} onChange={filtersUpdater.releaseId} title={'Release id'} />,
         releaseRule: () => (
            <RuleSelect
               stageId={stageId ?? ''}
               value={releaseRuleIdsAndDrafts}
               onChange={filtersUpdater.releaseRuleIdsAndDrafts}
            />
         ),
         status: () => <DeployTicketStatusSelect value={ticketStatuses} onChange={filtersUpdater.ticketStatuses} />,
      };

      const fieldList = Object.keys(filterFields) as Column[];
      const fields = fieldList.filter((name: Column) => !excludeColumns?.has(name));

      return (
         <div className={classNames(classes.filters, className)}>
            <div className={classes.fields} style={{ gridTemplateColumns: `repeat(${fields.length}, auto)` }}>
               {fields.map(name => filterFields[name]?.())}
            </div>

            <Button view={'action'} onClick={handleFilter} className={classes.action}>
               Filter
            </Button>
            <Button
               view={'outlined'}
               onClick={handleResetFilters}
               disabled={!isUsingFilters}
               className={classes.action}
            >
               Reset
            </Button>
         </div>
      );
   },
);

/**
 * Разбор значений фильтра по релизным правилам и драфтам, **ручное определение типов**
 */
function getReleaseRuleIdsAndDrafts<T extends Set<string>>(values: T): T & ReleaseRuleIdsAndDrafts {
   return new Set([...values.values()].filter(e => e === draftLabel || e.startsWith('rule:'))) as T &
      ReleaseRuleIdsAndDrafts;
}

/**
 * Разбор значений фильтра по статусам деплойных тикетов, **ручное определение типов**
 */
function getTicketStatuses<T extends Set<string>>(values: T): T & Set<TicketStatus> {
   return new Set([...values.values()].filter(e => (OnlyTicketStatuses as Set<string>).has(e))) as T &
      Set<TicketStatus>;
}

export function parseFiltersFromUrl(search: string): DeployTicketFilters {
   const parsed = fromQuery(search) as Partial<UrlFilters>;

   const statuses = queryValueToSet(parsed.ticketStatuses);

   const filters: DeployTicketFilters = {
      releaseRuleIdsAndDrafts: getReleaseRuleIdsAndDrafts(queryValueToSet(parsed.releaseRuleIdsAndDrafts)),
      ticketStatuses: statuses.size > 0 ? getTicketStatuses(statuses) : new Set(ActiveTicketStatuses),
   };

   const { stageId, ticketId, releaseId } = parsed;

   if (stageId) {
      filters.stageId = stageId;
   }
   if (ticketId) {
      filters.ticketId = ticketId;
   }
   if (releaseId) {
      filters.releaseId = releaseId;
   }

   return filters;
}

DeployTicketFiltersView.displayName = 'DeployTicketFiltersView';
