import { faDocker } from '@fortawesome/free-brands-svg-icons';
import { faRobot } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { YCSelect, YCSelectItemsGroup } from '@yandex-data-ui/common';
import { ButtonLink, classNames } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react';
import ReactDOM from 'react-dom';
import { useDispatch, useSelector } from 'react-redux';

import { urlBuilder } from '../../../../../models';
import { DeployTicketFilters, ReleaseRule, ReleaseRuleType } from '../../../../../models/ui';
import { fetchReleaseRules, RootState, selectRulesByStage, useStage } from '../../../../../redux';

import classes from './RuleSelect.module.scss';

type Value = DeployTicketFilters['releaseRuleIdsAndDrafts'];
type ValueItem = Value extends Set<infer X> ? X : never;

interface Props {
   onChange: (value: Value) => void;
   value: Value;
   stageId: string;
   width?: number;
}

export const RuleSelect: React.FC<Props> = React.memo(({ onChange, value, stageId, width }) => {
   const [flag, setFlag] = useState(false);
   const [renderVersion, setRenderVersion] = useState(0);

   const { stage } = useStage(stageId);
   const canAddSandboxRule = stage ? stage.sandboxTargets.length > 0 : false;
   const canAddDockerRule = stage ? stage.dockerTargets.length > 0 : false;

   const dispatch = useDispatch();

   const handleChange = useCallback((v: ValueItem[]) => onChange(new Set(v)), [onChange]);
   const items = useMemo(() => [...value.values()], [value]);

   useEffect(() => {
      dispatch(fetchReleaseRules(stageId));
   }, [dispatch, stageId]);

   const rulesSelector = useCallback((s: RootState) => selectRulesByStage(s, stageId), [stageId]);
   const rules = useSelector(rulesSelector);

   const getRobotIcon = useCallback(
      (rule: ReleaseRule) =>
         rule.autocommit ? (
            <FontAwesomeIcon className={classes.icon} icon={faRobot} title={'Autocommit is enabled'} />
         ) : null,
      [],
   );

   const getTypeIcon = useCallback(
      (rule: ReleaseRule) =>
         rule.type === ReleaseRuleType.Sandbox ? (
            <span className={classNames(classes.icon, classes.sandbox)} title={'Sandbox release rule'} />
         ) : (
            <FontAwesomeIcon
               className={classNames(classes.icon, classes.docker)}
               icon={faDocker}
               title={'Docker release rule'}
            />
         ),
      [],
   );

   const ticketTypeOptions: YCSelectItemsGroup[] = useMemo(
      () => [
         {
            groupTitle: 'Release rules',
            items: rules.map(rule => ({
               value: `rule:${rule.id}`,
               title: rule.id,
               url: urlBuilder.releaseRulePage(stageId, rule.id),
               icon: (
                  <div className={classes.ruleIconsContainer} style={{}}>
                     <div className={classes.ruleIcons}>
                        <div className={classes.ruleIcon}>{getTypeIcon(rule)}</div>
                        <div className={classes.ruleIcon}>{getRobotIcon(rule)}</div>
                     </div>
                  </div>
               ),
            })),
         },
         {
            groupTitle: 'Drafts',
            items: [
               {
                  value: `draft:all`,
                  title: 'All drafts',
               },
            ],
         },
      ],
      [getRobotIcon, getTypeIcon, rules, stageId],
   );

   const ticketTypeRef = useRef(null as Element | null);

   useEffect(() => {
      // каждую секунду проверяет появление всплывающего окна, при появлении запускает рендер с порталом
      const getElement = () => document.querySelector(`.yc-select-popup.${classes.ruleAndDraftSelect}`);
      ticketTypeRef.current = getElement();
      let timeout = 0;
      if (ticketTypeRef.current) {
         setRenderVersion(renderVersion + 1);
      } else {
         timeout = window.setTimeout(() => setFlag(!flag), 1000);
      }
      return () => {
         ticketTypeRef.current = getElement();
         if (timeout) {
            window.clearTimeout(timeout);
         }
      };
      // eslint-disable-next-line react-hooks/exhaustive-deps
   }, [flag]);

   const RuleButtons = ticketTypeRef.current
      ? ReactDOM.createPortal(
           <div className={classes.ruleButtons}>
              {canAddSandboxRule && (
                 <ButtonLink view={'action'} to={urlBuilder.releaseRuleCreate(stageId, ReleaseRuleType.Sandbox)}>
                    New Sandbox rule
                 </ButtonLink>
              )}

              {canAddDockerRule && (
                 <ButtonLink view={'action'} to={urlBuilder.releaseRuleCreate(stageId, ReleaseRuleType.Docker)}>
                    New Docker rule
                 </ButtonLink>
              )}
           </div>,
           ticketTypeRef.current,
        )
      : null;

   return (
      <div data-e2e={'ReleaseRule:Select'}>
         <YCSelect
            type={'multiple'}
            onUpdate={handleChange as (v: string[]) => void}
            value={items as string[]}
            items={ticketTypeOptions}
            showSearch={true}
            popupWidth={400} // #DEPLOY-4099 #DATAUI-776
            showItemIcon={true}
            showItemMeta={true}
            popupClassName={classes.ruleAndDraftSelect}
            placeholder={'Select or create rule'}
            controlWidth={width ?? 180}
         />
         {RuleButtons}
      </div>
   );
});

RuleSelect.displayName = 'RuleSelect';
