import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react';

import {
   CheckboxSelection,
   CheckboxSelectionItem,
   Loader,
   modalService,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-cloud/uikit';
import { faCheck, faArrowAltFromLeft, faTimes, IconDefinition, faDownload } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { map as mapOp, takeUntil } from 'rxjs/operators';
import { Observable, timer } from 'rxjs';

import { DeployTicket, MutableTicketStatuses, OptionalDeployTicketColumn, TicketAction } from '../../../models/ui';
import { FixedBlock } from '../../lib';
import { noop, pluralNumber } from '../../../utils';

import { DeployTicketRow } from '../DeployTicketRow/DeployTicketRow';
import { DeployTicketHead } from '../DeployTicketHead/DeployTicketHead';
import { DeployTicketSizeContainer } from '../DeployTicketSizeContainer/DeployTicketSizeContainer';
import { DeployTicketMassActionModal } from '../DeployTicketMassActionModal/DeployTicketMassActionModal';

import classes from './DeployTicketList.module.css';
import { DeployTicketPageType } from '../model';
import { useMassActionMessage } from '../hooks/useMassActionMessage';

const actionIcons: Record<TicketAction, IconDefinition> = {
   [TicketAction.Approve]: faCheck,
   [TicketAction.Disapprove]: faTimes,
   [TicketAction.Commit]: faDownload,
   [TicketAction.Skip]: faArrowAltFromLeft,
};

interface Props {
   pageType: DeployTicketPageType;
   tickets: DeployTicket[];
   withSelection?: boolean;
   loadAllIds?: Observable<Set<string>>;
}

/**
 * для разных страниц скрываем соответствующие поля
 */
const excludeColumnsByType: { [type in DeployTicketPageType]?: Set<OptionalDeployTicketColumn> } = {
   stage: new Set(['stage']),
   release: new Set(['release']),
};

// порядок важен
const actionNames = [TicketAction.Skip, TicketAction.Disapprove, TicketAction.Approve, TicketAction.Commit];

/**
 * Только список тикетов, который уже известен.
 * Сюда же входит обработка массовых операций.
 * Для пустого списка этот компонент возвратит null, обработка должна быть на уровень выше
 */
export const DeployTicketList: React.FC<Props> = React.memo(
   ({
      pageType,
      tickets,
      withSelection = false,
      loadAllIds = timer(5000).pipe(mapOp(() => new Set<string>('test'))),
   }) => {
      const excludeColumns: Set<OptionalDeployTicketColumn> = useMemo(
         () => excludeColumnsByType[pageType] ?? new Set(),
         [pageType],
      );
      const dismounted = useDismounted();

      // выделение только видимых тикетов
      const [isLocalSelection, setIsLocalSelection] = useState(true);

      // все тикеты в базе
      const [allIds, setAllIds] = useState<Set<string>>(new Set());
      // их загрузка
      const [loadingAll, setLoadingAll] = useState(false);

      const selectAll = useCallback(() => {
         if (!loadAllIds) {
            return;
         }
         setLoadingAll(true);
         loadAllIds.pipe(takeUntil(dismounted)).subscribe(
            ids => {
               setAllIds(ids);
               setLoadingAll(false);
               setIsLocalSelection(false); // меняем только при успешной загрузке
            },
            (...e) => {
               console.dir(e);
               setLoadingAll(false);
            },
         );
      }, [dismounted, loadAllIds]);

      const selectLocal = useCallback(() => {
         setIsLocalSelection(true);
      }, []);

      // выделение тикетов
      const [selectedTicketIds, setSelectedTicketIds] = useState(new Set<string>());
      const handleSelectTickets = useCallback((ticketIds: Set<string>) => setSelectedTicketIds(ticketIds), []);

      // выделение отдельных патчей
      const [selectedPatchIds, setSelectedPatchIds] = useState<Map<string, Set<string>>>(new Map());

      // TODO: добавить общие действия
      const onSelectPatches = useCallback(
         (id: string) => (patchIds: Set<string>) => {
            setSelectedPatchIds(map => {
               map.set(id, patchIds);
               return new Map(map);
            });
         },
         [],
      );

      const ticketIds = isLocalSelection ? selectedTicketIds : allIds;

      const ticketSize = ticketIds.size;

      // оставляем только те, которые не подходят под общую выборку и не пустые
      const selectedPatchesTicketIds = useMemo(
         () =>
            [...selectedPatchIds.keys()].filter(
               ticketId => selectedPatchIds.get(ticketId)!.size > 0 && !ticketIds.has(ticketId),
            ),
         [selectedPatchIds, ticketIds],
      );

      const getPatchesSize = useCallback((ticketId: string) => selectedPatchIds.get(ticketId)?.size ?? 0, [
         selectedPatchIds,
      ]);
      const isSelectedPatches = useMemo(() => selectedPatchesTicketIds.some(ticketId => getPatchesSize(ticketId) > 0), [
         getPatchesSize,
         selectedPatchesTicketIds,
      ]);
      const patchSize = useMemo(
         () => selectedPatchesTicketIds.reduce((sum, ticketId) => sum + getPatchesSize(ticketId), 0),
         [getPatchesSize, selectedPatchesTicketIds],
      );

      const showMassActions = ticketSize > 0 || isSelectedPatches;

      // свободное пространство для массовых операций
      useEffect(() => {
         const root = document.getElementById('root')!; // FIXME: убрать хак, но как?
         if (showMassActions) {
            root.classList.add(classes.freeSpace);
         } else {
            root.classList.remove(classes.freeSpace);
         }
         return () => root.classList.remove(classes.freeSpace);
      }, [showMassActions]);

      // готовим тикеты к массовой операции, собирая вместе патчи и явно отмеченные
      const selectedTickets = useMemo(() => {
         const map: Map<string, Set<string>> = new Map();
         for (const id of selectedPatchesTicketIds) {
            map.set(id, selectedPatchIds.get(id) ?? new Set());
         }
         for (const id of ticketIds.keys()) {
            map.set(id, new Set());
         }
         return map;
      }, [selectedPatchIds, selectedPatchesTicketIds, ticketIds]);

      // после массовой операции отменяем выделение
      const selectionRef = useRef(noop);
      const clearSelection = useCallback(() => {
         setSelectedTicketIds(new Set());
         setSelectedPatchIds(new Map());
         // все id также очищаем
         setLoadingAll(false);
         setIsLocalSelection(true);
         setAllIds(new Set());

         // сброс выделения на странице
         selectionRef.current();
      }, []);

      const getInitialMessage = useMassActionMessage({ pageType, selectedTickets });

      const showMassActionModal = useCallback(
         (action: TicketAction) => {
            modalService
               .open(DeployTicketMassActionModal, {
                  action,
                  initialMessage: getInitialMessage(action),
                  tickets: selectedTickets,
                  onStart: clearSelection,
               })
               .pipe(takeUntil(dismounted))
               .subscribe();
         },
         [clearSelection, dismounted, getInitialMessage, selectedTickets],
      );

      const idsForSelection = useMemo(
         () => tickets.filter(ticket => MutableTicketStatuses.has(ticket.status)).map(e => e.id),
         [tickets],
      );

      return (
         <DeployTicketSizeContainer>
            <CheckboxSelection allIds={idsForSelection} value={ticketIds} onChange={handleSelectTickets}>
               {({ isAllSelected, toggleAll, clear }) => {
                  selectionRef.current = clear;
                  return (
                     <div className={classes.list} data-test={'deploy-tickets-list'}>
                        <DeployTicketHead
                           mode={'list'}
                           toggleAll={() => {
                              toggleAll();
                              if (isAllSelected) {
                                 clearSelection();
                              }
                           }}
                           isAllSelected={isAllSelected}
                           excludeColumns={excludeColumns}
                        />

                        {tickets.map(({ id }) => (
                           <CheckboxSelectionItem key={id} id={id}>
                              {({ selected, toggle }) => (
                                 <DeployTicketRow
                                    selected={selected}
                                    toggle={() => {
                                       toggle(
                                          (new KeyboardEvent('click') as any) as React.KeyboardEvent<HTMLInputElement>,
                                       );
                                    }}
                                    mode={'list'}
                                    ticketId={id}
                                    excludeColumns={excludeColumns}
                                    selectedPatches={selectedPatchIds.get(id)}
                                    onSelectPatches={onSelectPatches(id)}
                                 />
                              )}
                           </CheckboxSelectionItem>
                        ))}
                     </div>
                  );
               }}
            </CheckboxSelection>
            {showMassActions && (
               <FixedBlock position={'bottom'} className={classes.massActions}>
                  <div>
                     <Button size={'l'} view={'normal-contrast'} onClick={isLocalSelection ? selectAll : selectLocal}>
                        {isLocalSelection ? 'Select all in YP' : 'Select visible tickets only'}
                        {loadingAll && <Loader inline={true} />}
                     </Button>
                  </div>
                  <div>
                     Selected {ticketSize > 0 && <span>{pluralNumber(ticketSize, 'ticket', 'tickets')}</span>}
                     {isSelectedPatches && (
                        <span>
                           {ticketSize > 0 && ' and '}
                           {pluralNumber(patchSize, 'patch', 'patches')}
                        </span>
                     )}
                  </div>
                  <div className={classes.actions}>
                     {actionNames.map(action => (
                        <Button
                           key={action}
                           size={'l'}
                           view={'normal-contrast'}
                           onClick={() => showMassActionModal(action)}
                        >
                           <FontAwesomeIcon icon={actionIcons[action]} /> {action[0].toUpperCase()}
                           {action.slice(1)}
                        </Button>
                     ))}
                  </div>
                  <div>
                     <Button view={'flat-contrast'} onClick={clearSelection} size={'l'}>
                        <FontAwesomeIcon icon={faTimes} size={'2x'} className={classes.close} />
                     </Button>
                  </div>
               </FixedBlock>
            )}
         </DeployTicketSizeContainer>
      );
   },
);

DeployTicketList.displayName = 'DeployTicketList';
